import logging
import yaml
import os
import sandbox.projects.sandbox.resources as sb_resources
from sandbox import sdk2
from sandbox import common
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp

OWNER = "YC_BOOTSTRAP"
BOOTSTRAP_SECRET = "ycloud_dev_secrets"
CI_SECRETS = "ycloud_ci_secrets"

ZOIDBERG_SSH_KEY = "robot-zoidberg-ssh"
ROBOT_ZOIDBERG = "robot-zoidberg"

YC_TESTING_SSH_KEY = "robot_yc_testing_key"
YC_DOGFOOD_SSH_KEY = "robot_yc_dogfood_key"

EXAMPLES_DIR = "/usr/share/yc/ci/bootstrap/examples"


# ROBOT_SSH_KEY = "GUESTS_KEY"
# OWNER = "guest"
# ROBOT_NAME = ""


class YcBootstrapCiTask(sdk2.Task):
    """ Task, for execute yc-ci. """

    class Requirements(sdk2.Requirements):
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            "Environment container resource",
            default_value=978574915,
            resource_type=sb_resources.LXC_CONTAINER,
            platform="linux_ubuntu_16.04_xenial",
            required=True,
        )

        with sdk2.parameters.Group("VM parameters") as vm_param:
            prefix = sdk2.parameters.String("Prefix for VM", default="yc-ci", required=True)
            cores = sdk2.parameters.Integer("Cores", default="8", required=True)
            memory = sdk2.parameters.Integer("Memory", default="34359738368", required=True)
            cloud_name = sdk2.parameters.String("Cloud name", default="yandexcloud", required=True)
            folder_name = sdk2.parameters.String("Folder name", default="cloudvm", required=True)
            public_zone = sdk2.parameters.String("Public zone for create VM", default="on.cloud-preprod.yandex.net",
                                                 required=True)
            zones = sdk2.parameters.List("Zones", default=["ru-central1-a", "ru-central1-b", "ru-central1-c"],
                                         required=True)

        with sdk2.parameters.Group("CI parameters") as ci_param:
            api_url = sdk2.parameters.String("IAAS API url",
                                             default="https://iaas.private-api.cloud-preprod.yandex.net", required=True)
            iam_private_url = sdk2.parameters.String("Identity private API url",
                                                     default="http://identity.private-api.cloud-preprod.yandex.net:4336",
                                                     required=True)
            ui_url = sdk2.parameters.String("UI url", default="console-preprod.cloud.yandex.net", required=True)
            pool_size = sdk2.parameters.Integer("Pool size (size of maximum VMS in cluster)", default=75, required=True)
            gc_min_age = sdk2.parameters.String("Min age of VM (minutes)", default="1440m", required=True)
            gc_max_age = sdk2.parameters.String("Max age of VM (minutes)", default="1560m", required=True)
            log_level = sdk2.parameters.String("Log level for task", default="ERROR", required=True)
            log_format = sdk2.parameters.String("Log format for task", default="devel", required=True)
            ssl_verify = sdk2.parameters.Bool("SSL verify", default=False)
            emergency_mode = False

    def set_secrets(self, file_name, secret_name):
        try:
            logging.info("Get secret {}".format(secret_name))
            secrets = sdk2.Vault.data(OWNER, secret_name)
            with open(file_name, "w") as file_o:
                yaml.dump(yaml.load(secrets), file_o, default_flow_style=False)
                logging.info("creating secret file")
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Can't get secret {}".format(secret_name))

    def execute_shell_command(self, command, description, wait=True):
        with ProcessLog(self, logger=logging.getLogger(description)) as process_log:
            popen = sp.Popen(command, shell=True, stdin=sp.PIPE, stdout=process_log.stdout, stderr=process_log.stdout, cwd=os.getcwd())

            if not wait:
                return

            return popen.wait()

    def prepare_control_config(self):
        control_yaml = yaml.safe_dump({"instance_pool": {"gc_min_age": self.Parameters.gc_min_age,
                                                         "gc_max_age": self.Parameters.gc_max_age,
                                                         "size": self.Parameters.pool_size},
                                       "intranet_clients": {"ssl_verify": self.Parameters.ssl_verify},
                                       "log": {"format": self.Parameters.log_format,
                                               "level": self.Parameters.log_level},
                                       "emergency_mode": self.Parameters.emergency_mode},
                                      default_flow_style=False)
        with open("control.yaml", "w") as control_file:
            control_file.write(control_yaml)

        self.execute_shell_command("mv control.yaml /etc/yc/ci/", "mv control")

    def prepare_bootstrap_config(self):
        bootstrap_ci_yaml = yaml.safe_dump({
            "cluster_map": {"public_zone": self.Parameters.public_zone},
            "bootstrap": {"hosts": {"name_prefix": self.Parameters.prefix}},
            "endpoints": {"yandex_cloud": {
                "api_url": self.Parameters.api_url,
                "identity_private_api_url": self.Parameters.iam_private_url,
                "cloud_name": self.Parameters.cloud_name,
                "folder_name": self.Parameters.folder_name,
                "ui_url": self.Parameters.ui_url
            }
            },
            "instance_configuration": {
                "yandex_cloud": {
                    "cloudvm": {
                        "zones": self.Parameters.zones,
                        "resources": {
                            "memory": self.Parameters.memory,
                            "cores": self.Parameters.cores
                        }
                    }
                }
            }
        }, default_flow_style=False)
        with open("./bootstrap_ci.yaml", "w") as config_file:
            config_file.write(bootstrap_ci_yaml)

    def prepare_ssh_key(self, key_path, secret_name, key_name="id_rsa"):
        k = sdk2.ssh.Key(self, OWNER, secret_name)
        with k:
            logging.info("create ssh-key")
            with open(key_name, "w") as f:
                f.write(k.key)

        os.chmod(key_name, int('600', 8))
        self.execute_shell_command("mv {} {}/.ssh/".format(key_name, key_path), "rename ssh-key")
        self.execute_shell_command("ssh-add {}/.ssh/{}".format(key_path, key_name),
                                   "add ssh-key for {}".format(key_path))

    def install_yc_ci_packages(self, package_name="yc-ci"):
        self.execute_shell_command("apt-get update && apt-get download {}".format(package_name), "download packages")
        self.execute_shell_command("mv {}_*.deb {}.deb".format(package_name, package_name), "prepare packages")
        self.execute_shell_command("apt-get install -y --allow-downgrades {}/{}.deb".format(os.getcwd(), package_name),
                                   "install package")

    def on_execute(self):

        self.prepare_control_config()
        self.prepare_bootstrap_config()
        self.install_yc_ci_packages()
        self.install_yc_ci_packages("yc-bootstrap-tests")

        self.set_secrets("bootstrap_secrets.yaml", BOOTSTRAP_SECRET)
        self.set_secrets("secrets_ci.yaml", CI_SECRETS)
        self.prepare_ssh_key("/home/{}".format(ROBOT_ZOIDBERG), ZOIDBERG_SSH_KEY)
        self.prepare_ssh_key("/root", YC_TESTING_SSH_KEY, 'robot-yc-testing.key')
        self.prepare_ssh_key("/root", YC_DOGFOOD_SSH_KEY, 'robot-yc-dogfood.key')

        # Delete PYTHONPATH because it contains arcadia paths
        del os.environ['PYTHONPATH']

        command = ["/usr/bin/yc-ci",
                   "--package", "./yc-ci.deb",
                   "--bootstrap-template",
                   "{}/cloudvm.yaml".format(EXAMPLES_DIR), "./bootstrap_secrets.yaml", "./bootstrap_ci.yaml",
                   "--rackmap", "{}/shared/rackmap.yaml".format(EXAMPLES_DIR),
                   "--secret-conf", "./secrets_ci.yaml",
                   "--ssh-user", ROBOT_ZOIDBERG,
                   "--project", "yc-ci",
                   "--branch", "refs/heads/master",
                   "--bit-bucket"]
        status = self.execute_shell_command(" ".join(command), "run-ci")
        logging.info("Status {}".format(status))

        if status != 0:
            raise Exception("Can't complete yc-ci process, see log for details")
