import os
import json
import logging
import string
import random

import xml.dom.minidom as xml_parser

import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common import utils
from sandbox.projects import resource_types
from sandbox.projects.common.build import parameters, YaPackage


TARBALL_WITH_CHECKOUT = YaPackage.TARBALL + "_with_checkout"


class DontPublishDebianPackagesParameter(sdk_parameters.SandboxBoolParameter):
    name = 'dont_publish_debian_packages'
    description = 'Do not publish debian packages'
    required = False
    default_value = False


class YaPackageAcceptanceTask(SandboxTask):
    type = 'YA_PACKAGE_ACCEPTANCE'
    input_parameters = [
        DontPublishDebianPackagesParameter,
        parameters.UseDevVersion,
    ] + parameters.get_arcadia_params()

    def on_execute(self):
        if 'child_tasks_ids' not in self.ctx:
            self.ctx['child_tasks_ids'] = self._create_package_tasks()
            self.wait_tasks(
                tasks=self.ctx['child_tasks_ids'].values(),
                statuses=tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK),
                wait_all=True,
            )
        else:
            result = self._check_package_tasks(self.ctx['child_tasks_ids'])

            report_path = "results2.json"
            result.dump(report_path)
            attributes = {
                'ttl': 30,
            }
            self._create_resource("json report", report_path, resource_types.TEST_ENVIRONMENT_JSON_V2, complete=1, attrs=attributes)
            self.set_info(str(result), do_escape=False)
            if self.ctx.get('fail_on_any_error') and not result.passed:
                raise SandboxTaskFailureError("Acceptance failed")

    def _create_package_tasks(self):
        packages = [
            "sandbox/projects/YaPackageAcceptance/packages/package_new.json",
            "sandbox/projects/YaPackageAcceptance/packages/package_with_relative.json",
        ]
        child_tasks_ids = {}

        # create tar packages
        child_tasks_ids[YaPackage.TARBALL] = self._create_ya_package_subtask(
            packages, YaPackage.TARBALL, "build package for acceptance task: {}".format(YaPackage.TARBALL)
        ).id

        if not self.ctx[parameters.ArcadiaPatch.name]:
            child_tasks_ids[TARBALL_WITH_CHECKOUT] = self._create_ya_package_subtask(
                packages, YaPackage.TARBALL, "build package for acceptance task: {}".format(TARBALL_WITH_CHECKOUT), with_checkout=True,
            ).id

        if not self.ctx[DontPublishDebianPackagesParameter.name]:
            publish_to = "search-test"
        else:
            publish_to = None

        child_tasks_ids["with_small_tests"] = self._create_ya_package_subtask(
            ["sandbox/projects/YaPackageAcceptance/packages/package_with_tests.json"],
            YaPackage.TARBALL, "build package for acceptance task: with_small_tests", run_tests=True
        ).id

        child_tasks_ids["with_long_tests"] = self._create_ya_package_subtask(
            ["sandbox/projects/YaPackageAcceptance/packages/package_with_tests.json"],
            YaPackage.TARBALL, "build package for acceptance task: with_long_tests", run_tests=True, run_all_tests=True
        ).id

        # create debian packages
        child_tasks_ids[YaPackage.DEBIAN] = self._create_ya_package_subtask(
            packages, YaPackage.DEBIAN,
            "build package for acceptance task: {}".format(YaPackage.DEBIAN),
            publish_to=publish_to,
        ).id

        return child_tasks_ids

    def _create_ya_package_subtask(self, packages, package_format, description, publish_to=None, with_checkout=False, run_tests=False, run_all_tests=False):
        ya_pkg_ctx = {
            "acceptance_testing": True,
            parameters.ArcadiaUrl.name: self.ctx[parameters.ArcadiaUrl.name],
            parameters.ArcadiaPatch.name: self.ctx[parameters.ArcadiaPatch.name],
            'use_ya_dev': self.ctx[parameters.UseDevVersion.name],
            YaPackage.PackagesParameter.name: ";".join(packages),
            YaPackage.PackageTypeParameter.name: package_format,
            YaPackage.ResourceTypeParameter.name: resource_types.YA_PACKAGE.name,
        }
        if publish_to:
            ya_pkg_ctx[YaPackage.PublishPackageParameter.name] = True
            ya_pkg_ctx[YaPackage.PublishToParameter.name] = publish_to
            ya_pkg_ctx[YaPackage.KeyUserParameter.name] = "robot-admins"
        else:
            ya_pkg_ctx[YaPackage.PublishPackageParameter.name] = False

        if with_checkout:
            ya_pkg_ctx[parameters.CheckoutParameter.name] = True

        if run_tests:
            ya_pkg_ctx[YaPackage.RunTestsParameter.name] = True
            if run_all_tests:
                ya_pkg_ctx[YaPackage.RunLongTestsParameter.name] = True

        return self.create_subtask(
            task_type='YA_PACKAGE', input_parameters=ya_pkg_ctx, description=description, inherit_notifications=True,
        )

    def _check_package_tasks(self, child_tasks_ids):
        utils.check_if_tasks_are_ok(child_tasks_ids.values())
        result = _CheckResult()
        self._check_tar_tasks(child_tasks_ids[YaPackage.TARBALL], result)
        if TARBALL_WITH_CHECKOUT in child_tasks_ids:
            self._check_tar_tasks(child_tasks_ids[TARBALL_WITH_CHECKOUT], result)
        self._check_debian_tasks(child_tasks_ids[YaPackage.DEBIAN], result)
        self._check_tasks_with_tests(child_tasks_ids["with_small_tests"], child_tasks_ids["with_long_tests"], result)
        return result

    def _check_tasks_with_tests(self, small_tests_task_id, all_tests_task_id, result):
        def get_test_results(task_id):
            resources = channel.sandbox.list_resources(
                task_id=task_id,
                resource_type=resource_types.TASK_LOGS.name
            )
            path = self.sync_resource(resources[0])
            return self._parse_junit(os.path.join(path, "junit_yandex-package-test.xml"))

        def verify_tests(expected, actual):
            logging.info("Expected: %s, actual: %s", expected, actual)
            for project in expected:
                assert project in actual, "Project {} is expected to be present in {}".format(project, actual)
                for test, test_result in expected[project].items():
                    assert test in actual[project], "Test {} is expected to be prsent in {}".format(test, actual[project])
                    assert test_result == actual[project][test], "{}: expected {} != actual {}".format(test, test_result, actual[project][test])

        passed = True
        comment = None
        expected_tests = {u'devtools/ya/test/tests/test_size/data/py': {u'test_small.py.test': {'comment': None, 'status': 1}}}
        try:
            verify_tests(expected_tests, get_test_results(small_tests_task_id))
        except Exception as e:
            passed = False
            comment = str(e)
        result.add_test_record("with_long_tests", passed, comment, "https://sandbox.yandex-team.ru/task/{}".format(small_tests_task_id))

        passed = True
        comment = None
        expected_tests = {
            u"devtools/ya/test/tests/test_size/data/py": {u"test_small.py.test": {'comment': None, 'status': 1}},
            u'devtools/ya/test/tests/test_size/data/py/medium': {u'test_medium.py.test': {'comment': None, 'status': 1}},
            u'devtools/ya/test/tests/test_size/data/py/fat': {u'test_fat.py.test': {'comment': None, 'status': 1}}
        }
        try:
            verify_tests(expected_tests, get_test_results(all_tests_task_id))
        except Exception as e:
            passed = False
            comment = str(e)
        result.add_test_record("with_small_tests", passed, comment, "https://sandbox.yandex-team.ru/task/{}".format(all_tests_task_id))

    def _check_tar_tasks(self, task_id, result):
        expected_tars = {'yandex-package-test-new.1.tar.gz', 'yandex-package-test-relative.1.tar.gz'}
        resources = channel.sandbox.list_resources(
            task_id=task_id,
            resource_type=resource_types.YA_PACKAGE.name
        )
        logging.info("Tar task resources: %s", resources)
        actual_file_names = [r.file_name for r in resources]
        logging.info("file_names: %s", actual_file_names)
        passed = True
        comment = None
        if set(actual_file_names) != expected_tars:
            passed = False
            comment = "Expected resource file names {}, got {}".format(expected_tars, actual_file_names)
        result.add_test_record("test_tar_format", passed, comment, "https://sandbox.yandex-team.ru/task/{}".format(task_id))

    def _check_debian_tasks(self, task_id, result):
        expected_tars = {'yandex-package-test-new.1.tar.gz', 'yandex-package-test-relative.1.tar.gz'}
        resources = channel.sandbox.list_resources(
            task_id=task_id,
            resource_type=resource_types.YA_PACKAGE.name
        )
        logging.info("Tar task resources: %s", resources)
        actual_file_names = [r.file_name for r in resources]
        logging.info("file_names: %s", actual_file_names)
        passed = True
        comment = None
        if set(actual_file_names) != expected_tars:
            passed = False
            comment = "Expected resource file names {}, got {}".format(expected_tars, actual_file_names)
        result.add_test_record("test_debian_format", passed, comment, "https://sandbox.yandex-team.ru/task/{}".format(task_id))

        # release to check on_release dupload
        if self.ctx[DontPublishDebianPackagesParameter.name]:
            self.create_release(task_id)

    def _parse_junit(self, junit_path, fill_suites=False):
        with open(junit_path) as junit:
            report = xml_parser.parseString(junit.read())

        tests = {}
        for suite_el in report.getElementsByTagName("testsuite"):
            subtests = {}
            for test_case_el in suite_el.getElementsByTagName("testcase"):
                test_name = test_case_el.attributes["name"].value

                if test_case_el.getElementsByTagName("failure"):
                    test_comment = test_case_el.getElementsByTagName("failure")[0].childNodes[0].wholeText
                    test_status = 3
                elif test_case_el.getElementsByTagName("skipped"):
                    test_comment = test_case_el.getElementsByTagName("skipped")[0].childNodes[0].wholeText
                    test_status = -100
                else:
                    test_comment = None
                    test_status = 1

                if test_name in subtests:
                    raise Exception("Duplicating test '{}' found in suite '{}'".format(test_name, suite_el.attributes["name"].value))
                subtests[test_name] = {
                    "comment": test_comment,
                    "status": test_status
                }
            if fill_suites:
                suite_dict = {}
                for attr in ["failures", "name", "skipped", "tests", "time"]:
                    suite_dict[attr] = suite_el.attributes[attr].value
                subtests["suite"] = suite_dict
            suite_name = suite_el.attributes["name"].value
            if suite_name not in tests:
                tests[suite_name] = {}
            tests[suite_name].update(subtests)
        return tests


class _CheckResult(object):

    def __init__(self):
        self._report = []

    def add_test_record(self, subtest_name, passed, comment, log_link):
        record = {
            "owners": {
                "logins": [
                    "dmitko"
                ],
                "groups": [
                    "yatool"
                ]
            },
            "toolchain": "default-linux-x86_64-release",
            "uid": "oh5MrWBg6Y8zOpf-xBU5Kw",
            "links": {
                "log": [
                    log_link
                ]
            },
            "path": "sandbox/projects/YaPackageAcceptance",
            "type": "test",
            "id": subtest_name,
            "name": "acceptance",
            "subtest_name": subtest_name
        }
        if passed:
            record["status"] = "OK"
        else:
            record["status"] = "FAILED"
            record["error_type"] = "REGULAR"
            record["rich-snippet"] = comment
        record["uid"] = uniq_string_generator(8)()
        self._report.append(record)

    def dump(self, path):
        with open(path, "w") as fp:
            json.dump(self._report, fp, indent=4)

    @property
    def passed(self):
        for record in self._report:
            if record["status"] != "OK":
                return False
        return True

    def __str__(self):
        lines = []
        for record in self._report:
            lines.append("<strong>[{}] {}</strong>".format(record["status"], record["subtest_name"]))
            if "rich-snippet" in record:
                lines.append(record["rich-snippet"])
            lines.append("Task: <a href='{link}' target='_blank'>{link}</a>".format(link=record["links"]["log"][0]))
        return "<br>".join(lines)


def uniq_string_generator(size=6, chars=string.ascii_lowercase + string.digits):
    return lambda: ''.join(random.choice(chars) for _ in range(size))


__Task__ = YaPackageAcceptanceTask
