# coding: utf-8

import os
import re
import json
import logging

import sandbox.common.types.client as ctc

from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common import constants

import sandbox.sandboxsdk as ssdk


class YaTestDescriptionUpdater(ArcadiaTask):

    type = 'YA_TEST_DESCRIPTION_UPDATER'
    te_job_name_limit = 100
    te_job_name_prefix = "FAT"
    regexp_valid_te_name = re.compile(r"[^\w/-]")

    client_tags = ctc.Tag.HDD

    class TargetTestSize(ssdk.parameters.SandboxStringParameter):
        name = 'test_size'
        description = 'Test size'
        default_value = ''

    class TestType(ssdk.parameters.SandboxStringParameter):
        name = 'test_type'
        description = 'Test type'
        default_value = ''

    class Target(ssdk.parameters.SandboxStringParameter):
        name = 'target_path'
        description = 'Target path (source_root by default)'
        default_value = ''

    class JobPrefix(ssdk.parameters.SandboxStringParameter):
        name = 'job_prefix'
        description = 'TE job prefix'
        default_value = 'TEST'

    class BuildFlags(ssdk.parameters.SandboxStringParameter):
        name = 'build_flags'
        description = "Flags (without '-D' prefix) (space separated)"
        default_value = ''

    class TargetDescriptionFile(ssdk.parameters.SandboxStringParameter):
        name = 'description_path'
        description = "Path to the description"
        required = True
        default_value = ''

    class SkipTags(ssdk.parameters.SandboxStringParameter):
        name = 'skip_tags'
        description = 'Skip tags (space separated)'
        default_value = 'ya:manual ya:not_autocheck'

    class DryRun(ssdk.parameters.SandboxBoolParameter):
        name = 'dry_run'
        description = 'Dry run (do not commit changes)'
        default_value = True

    input_parameters = ArcadiaTask.input_parameters + [
        Target,
        TargetTestSize,
        TestType,
        JobPrefix,
        BuildFlags,
        TargetDescriptionFile,
        SkipTags,
        DryRun,
    ]

    def do_execute(self):
        # TODO get rid of fixed binary version
        resource_id = 2863997659
        logging.info("Downloading test_description_updater tool (%d)", resource_id)
        # XXX bruh
        tdu_path = self.sync_resource(resource_id)
        logging.info("Getting arcadia")
        source_root = self.get_arcadia_src_dir()
        if self.ctx.get(constants.ARCADIA_PATCH_KEY):
            logging.info("Applying patch")
            ssdk.svn.Arcadia.apply_patch(source_root, self.ctx[constants.ARCADIA_PATCH_KEY], self.abs_path())

        # XXX extra logging for debug purposes
        logging.info("Obtaining arcadia stamp")
        proc = ssdk.process.run_process(["ls", "-Rl", "--time-style=posix-long-iso"], outs_to_pipe=True, wait=False, work_dir=source_root)
        stdout, stderr = proc.communicate()
        with open(self.log_path("arcadia_stamp.txt"), "w") as afile:
            afile.write(">>STDOUT:\n{}\n>>STDERR:\n{}".format(stdout, stderr))

        output = self.log_path("description.json")
        description_filename = os.path.join(source_root, self.ctx["description_path"])

        self.dump_tests_description(tdu_path, source_root, description_filename, output)
        with open(output) as afile:
            test_desc = json.load(afile)
        with open(description_filename) as afile:
            committed_desc = json.load(afile)

        if committed_desc.get("tests") == test_desc.get("tests"):
            logging.info("No changes in test description found - no action is required")
        elif self.ctx["dry_run"]:
            logging.info("Dry run option specified - no action is required")
        else:
            commit_msg = 'Updated tests description by sandbox task #{} SKIP_CHECK'.format(self.id)
            logging.info("Going to commit updated tests descriptions with message: %s", commit_msg)
            with open(description_filename, "w") as afile:
                json.dump(test_desc, afile, indent=4, sort_keys=True)
            ssdk.svn.Arcadia.commit(description_filename, commit_msg, 'zomb-sandbox-rw')

        error_msg = ""
        # XXX
        # We want to get notifications about missing tests.
        # Generally it's removal or moving of the test, however, there was heisenbug regression of ymake,
        # when tests were missing without visible reasons.
        if test_desc["meta"]["missing_tests"]:
            error_msg += "Missing tests found: {}\n".format(json.dumps(test_desc["meta"]["missing_tests"], indent=4, sort_keys=True))
        if test_desc["meta"]["duplicates"] and test_desc["meta"]["duplicates"] != committed_desc["meta"].get("duplicates", []):
            error_msg += "Duplicates found: {}\n".format(json.dumps(test_desc["meta"]["duplicates"], indent=4, sort_keys=True))
        if error_msg:
            raise Exception(error_msg)

    def dump_tests_description(self, tdu_path, source_root, description_file, output):
        logging.info("Obtaining tests description")
        cmd = [
            tdu_path,
            "--job-prefix", self.ctx["job_prefix"],
            "--output", output,
            "--description-file", description_file,
            "--revision", ssdk.svn.Arcadia.info(source_root)["commit_revision"],
            "--source-root", source_root,
            "--ya-tool", os.path.join(source_root, "ya"),
            "--verbose",
        ]
        if self.ctx["test_size"]:
            cmd += ["--test-size", self.ctx["test_size"].strip()]
        if self.ctx["skip_tags"]:
            skip_tags = self.ctx["skip_tags"].strip(" ")
            skip_tags = re.sub(r"\s{2,}", " ", skip_tags)
            cmd += ["--skip-tags"] + skip_tags.split(" ")
        if self.ctx["build_flags"]:
            cmd += ["--flags"] + self.ctx["build_flags"].strip().split(" ")
        if self.ctx["target_path"]:
            cmd += ["--target", self.ctx["target_path"].strip()]
        if self.ctx["test_type"]:
            cmd += ["--test-type", self.ctx["test_type"].strip()]

        logfile = self.log_path("tests_description_updater")
        with open(logfile + ".out.log", "w") as stdout, open(logfile + ".err.log", "w", buffering=0) as stderr:
            proc = ssdk.process.run_process(cmd, log_prefix='tdu', stdout=stdout, stderr=stderr, wait=True)
        if proc.returncode != 0:
            raise Exception("Bad return code %d. See logs for more information.", proc.returncode)


__Task__ = YaTestDescriptionUpdater
