# -*- coding: utf-8 -*-

import os
import copy
import json
import logging
import collections

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel

import sandbox.common.types.client as ctc
import sandbox.common.platform as sb_platform
import sandbox.projects.common.constants as consts
from sandbox.projects.common.arcadia import resources as sdk_resources
from sandbox.projects.common.build import dependency_banner
from sandbox.projects.common.build.sdk import sdk_compat
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects import resource_types

from sandbox.projects.autocheck.lib.deprecated import dist_priority as dp

import merger

SEMIDIST_OWNERS = ['YT', 'YT_ROBOT']


class TestsRetriesCount(parameters.SandboxIntegerParameter):
    name = consts.SUBTASK_COUNT
    description = 'Subtask count'
    required = True
    default_value = 3


class NativeTargetsParameter(parameters.SandboxBoolParameter):
    name = 'native_targets'
    description = 'Native build targets'
    required = False
    default_value = False


class NativeBuildPathParameter(parameters.SandboxStringParameter):
    name = 'native_build_path'
    description = 'Native build path'
    required = False
    default_value = ''


class ChildTaskSandboxTags(parameters.SandboxStringParameter):
    name = 'child_task_sandbox_tags'
    separator = ','
    description = 'Sandbox tags for child task, %s-separated:' % separator


CTX_CHILD_TASKS_IDS = 'child_tasks_ids'
MAPSMOBI_TAG = 'ya:exotic_platform_use_maps_mobile_hosts'
PORTOD_TAG = 'sb:portod'
PRIVILEGED_TAG = 'ya:privileged'


def get_input_params():
    params = collections.OrderedDict()
    for p in [
        TestsRetriesCount,
        ChildTaskSandboxTags,
        NativeTargetsParameter,
        NativeBuildPathParameter,
     ] + YaMakeTask.input_parameters:
        params[p.name] = p

    # some parameters are unwanted to set, some will be hardcoded
    for unwanted_name in [
        'result_rt', 'result_single_file', 'arts_source', 'arts',
        dependency_banner.BANNED_DEPS_KEY,
        dependency_banner.WHITELISTED_DEPS_KEY,
        dependency_banner.CHECK_DEPS_KEY,
        dependency_banner.BAN_UT_KEY,
        consts.FORCE_BUILD_DEPENDS,
        consts.BUILD_BUNDLE_KEY,
        consts.USE_YA_BLOAT,
        consts.JUNIT_REPORT,
        consts.ALLURE_REPORT,
        consts.ALLURE_REPORT_TTL,
        consts.REPORT_TESTS_ONLY,
        consts.COVERAGE,
        consts.JAVA_COVERAGE,
        consts.COVERAGE_PREFIX_FILTER,
        consts.COVERAGE_EXCLUDE_REGEXP,
        consts.PGO_ADD,
        consts.PGO_USE,
        consts.PGO_MERGE_TIMEOUT,
        consts.TESTS_REQUESTED,
        consts.TEST_FAILURE_CODE_KEY,
        consts.SONAR,
        consts.SONAR_OPTIONS,
        consts.SONAR_PROJECT_FILTER,
        consts.SONAR_DEFAULT_PROJECT_FILTER,
        consts.USE_SYSTEM_PYTHON,
        consts.CHECK_RETURN_CODE,
    ]:
        if unwanted_name in params:
            del params[unwanted_name]
    return params.values()


class YaTestParentTask(YaMakeTask):
    """
    DEPRECATED
    Таска не поддерживается и не развивается. Используейте YA_TEST_PARENT_2
    Прогон тестов: запуск тестов в заданном числе дочерних задач с объединением результатов
    """

    type = 'YA_TEST_PARENT'
    input_parameters = get_input_params()
    client_tags = (
        (ctc.Tag.GENERIC | ctc.Tag.Group.OSX) &
        (~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_XENIAL | ctc.Tag.LINUX_BIONIC)
    )
    cores = 1

    def on_execute(self):
        if CTX_CHILD_TASKS_IDS not in self.ctx:
            self.ctx[CTX_CHILD_TASKS_IDS] = self._create_subtasks()
            self.wait_tasks(
                tasks=self.ctx[CTX_CHILD_TASKS_IDS],
                statuses=tuple(self.Status.Group.FINISH) + tuple(self.Status.Group.BREAK),
                wait_all=True,
            )
        else:
            # from projects.common import utils
            # utils.check_if_tasks_are_ok(self.ctx['child_tasks_ids'])

            subtask_results = {
                "https://sandbox.yandex-team.ru/task/{}".format(task_id): self._get_subtask_tests_results(
                    task_id) for task_id in self.ctx['child_tasks_ids']
            }
            try:
                task_id = self.ctx['child_tasks_ids'][-1]
                subtask_stat_path = self._get_subtask_stats(task_id)
            except Exception:
                subtask_stat_path = None
                logging.exception("getting subtask statistics failed")

            quorum_count = 1  # int(self.ctx[consts.SUBTASK_COUNT]) - 1
            logging.debug("Will merge with quorum_count=%s", quorum_count)

            if self.ctx.get(NativeTargetsParameter.name, False):
                test_results = merger.native_build_merge(
                    subtask_results,
                    {x.strip() for x in self.ctx['targets'].split(';') if x},
                    self.ctx[NativeBuildPathParameter.name],
                )
            else:
                test_results = merger.merge(subtask_results, quorum_count=quorum_count)

            result = {
                "results": test_results,
                "static_values": {},
            }
            if subtask_stat_path:
                try:
                    self._add_statistics_to_result(result, subtask_stat_path)
                except Exception:
                    logging.exception("add_statistics_to_result failed")
            report_path = "results.json"
            with open(report_path, "w") as report:
                json.dump(result, report)

            self._create_resource("json report", report_path, resource_types.TEST_ENVIRONMENT_JSON_V2, complete=1, attrs={'ttl': 30})

            streaming_link = self.ctx.get(consts.STREAMING_REPORT_URL, None)
            streaming_check_id = self.ctx.get(consts.STREAMING_REPORT_ID, None)
            if streaming_link is not None and streaming_check_id is not None:
                self._report_streaming_results(streaming_link, streaming_check_id, report_path)

    def _create_subtasks(self):
        child_tasks_ids = []

        for i in range(int(self.ctx[consts.SUBTASK_COUNT])):
            child_tasks_ids.append(
                self._create_subtask("{} (Run #{})".format(self.descr, i)).id
            )

        return child_tasks_ids

    def _create_subtask(self, description):
        ctx = copy.copy(self.ctx)
        if self.ctx.get(NativeTargetsParameter.name):
            ctx[consts.TESTS_REQUESTED] = False
            # Test result will be deduced in the YA_TEST_PARENT task, depending on build and configure successfulness
            # For more info see https://st.yandex-team.ru/CI-1259
            ctx[consts.REPORT_TESTS_ONLY] = False
        else:
            ctx[consts.TESTS_REQUESTED] = True
            ctx[consts.REPORT_TESTS_ONLY] = True
        ctx[consts.CHECK_RETURN_CODE] = False
        ctx[consts.STREAMING_REPORT_URL] = None
        ctx[consts.STREAMING_REPORT_ID] = None
        ctx[consts.DOWNLOAD_ARTIFACTS_FROM_DISTBUILD] = True
        ctx[consts.YA_YT_TOKEN_VAULT_OWNER] = self.owner

        expected_suite = json.loads(self.ctx.get("expected_test_info", "{}"))

        requirements = expected_suite.get("requirements", {})

        if requirements.get("container"):
            ctx[consts.SANDBOX_CONTAINER] = requirements["container"]

        tags = expected_suite.get("tags", [])
        sandbox_tags = []
        for tag in self.ctx.get(ChildTaskSandboxTags.name, "").split(ChildTaskSandboxTags.separator):
            if tag:
                sandbox_tags.append(tag)
        for tag in tags:
            if tag.startswith("sb:"):
                sandbox_tags.append(tag[3:])

        max_restarts = None
        if MAPSMOBI_TAG in tags:
            ctx[consts.BUILD_SYSTEM_KEY] = consts.YA_MAKE_FORCE_BUILD_SYSTEM
            ctx[consts.USE_AAPI_FUSE] = True
            ctx[consts.USE_ARC_INSTEAD_OF_AAPI] = True
            ctx[consts.SANDBOX_TAGS] = 'CUSTOM_MAPS_MOBILE_DARWIN'
            ctx[consts.YA_YT_STORE] = True
            ctx[consts.YA_YT_STORE_CODEC] = 'zstd08_1'
            ctx[consts.YA_YT_PROXY] = 'hahn.yt.yandex.net'
            ctx[consts.YA_YT_DIR] = '//home/maps_mobile_build_cache'
            ctx[consts.YA_YT_PUT] = True
            ctx[consts.YA_YT_TOKEN_VAULT_NAME] = "robot-mapkit-ci-yt-token"
            ctx[consts.YA_YT_TOKEN_VAULT_OWNER] = "robot-mapkit-ci"
            ctx[consts.YA_YT_STORE_REFRESH_ON_READ] = True
        elif self._is_distbuild_allowed(tags):
            if self._is_force_distbuild(tags):
                sandbox_tags.append('MULTISLOT')
                ctx[consts.BUILD_SYSTEM_KEY] = consts.DISTBUILD_BUILD_SYSTEM
            else:
                ctx[consts.BUILD_SYSTEM_KEY] = consts.SEMI_DISTBUILD_BUILD_SYSTEM

            ctx[consts.RETRIABLE_DISPLACED_BUILD] = True
            ctx[consts.OUTPUT_ONLY_TESTS] = True
            # Add extra retries to overcome Distbuild's displace error
            ctx['max_restarts'] = max_restarts = 200  # 5h at least
            revision = ctx.get('revision')
            if ctx.get('is_precommit') and revision:
                ctx[consts.DISTBUILD_PRIORITY] = dp.get_dist_priority(-1, revision, dp.LARGE_TESTS_PRIORITY_DELTA)
        else:
            ctx[consts.OUTPUT_ONLY_TESTS] = False
            ctx[consts.BUILD_SYSTEM_KEY] = consts.YMAKE_BUILD_SYSTEM

        if sandbox_tags:
            ctx[consts.SANDBOX_TAGS] = "|".join(sandbox_tags)

        # SANDBOX-8482 Porto socket avaliable ONLY in privileged porto containers
        if PRIVILEGED_TAG in tags or PORTOD_TAG in tags:
            ctx[consts.PRIVILEGED] = True
            # privileged containers mount arcadia cache in read only mode and cannot update it
            ctx[consts.CHECKOUT] = True
            ctx[consts.CLEAR_BUILD_KEY] = True

        def normalize_size_req(req):
            # this normalizes size requirement size for trunk where all reqs became size in Gb
            # and branches, where sizes were in
            if req < 1000:
                return req * 1024 ** 3
            return req

        disk_usage_in_bytes = normalize_size_req(requirements.get("disk_usage", 10 if MAPSMOBI_TAG in tags else 100))
        ram_in_bytes = normalize_size_req(requirements.get("ram", 8))
        ram_disk_in_bytes = normalize_size_req(requirements.get("ram_disk", 0))

        if ram_disk_in_bytes:
            ctx[consts.RAM_DRIVE_SIZE] = ram_disk_in_bytes / 1024 ** 2  # ram_disk takes space in MB

        if MAPSMOBI_TAG in tags:
            platform = 'osx'
        elif requirements.get("container"):
            ctx[consts.SANDBOX_CONTAINER] = requirements["container"]
            platform = self.arch
        else:
            platform = 'linux_ubuntu_16.04_xenial'
            tag_platforms = set(sandbox_tags) & set(sb_platform.TAG_TO_PLATFORM.keys())
            if tag_platforms:
                platform = sb_platform.TAG_TO_PLATFORM[next(iter(tag_platforms))]

        env_vars = {}
        if requirements.get("sb_vault"):
            vaults_vars = requirements["sb_vault"].split(",")
            for var in vaults_vars:
                name, val = var.split("=", 1)
                env_vars[name] = '$(vault:{})'.format(val)

        if env_vars:
            ctx[consts.ENV_VARS] = " ".join(["{}={}".format(k, v) for k, v in env_vars.iteritems()])

        if requirements.get("dns") == "dns64":
            ctx[consts.DNS64_REQUIRED] = True

        for unwanted in [
            "__te_apiargs",
            "hosts_match_error",
            "client_tags_predicates",
            "streaming_check_id",
            "streaming_link",
        ]:
            if unwanted in ctx:
                del ctx[unwanted]

        subtask_type = 'YA_MAKE'
        if MAPSMOBI_TAG in tags:
            subtask_type = 'YA_MAKE_MAPS_MOBILE_OSX'
            if consts.ARCADIA_URL_KEY in ctx and (ctx[consts.ARCADIA_URL_KEY] or '').startswith('arcadia:/arc/trunk/arcadia@'):
                ctx[consts.ARCADIA_URL_KEY] = ctx[consts.ARCADIA_URL_KEY].replace('arcadia:/arc/trunk/arcadia@', 'arcadia-arc:/#r')

        return self.create_subtask(
            task_type=subtask_type,
            input_parameters=ctx,
            description=description,
            inherit_notifications=True,
            execution_space=disk_usage_in_bytes / 1024 ** 2,  # in MB
            ram=ram_in_bytes / 1024 ** 2,  # in MB
            model=self.model,
            arch=platform,
            tags=self.tags,
            max_restarts=max_restarts,
        )

    def _get_subtask_tests_results(self, task_id):
        task_object = channel.sandbox.get_task(task_id)
        logging.info("Child task %s status: %s", task_id, task_object.new_status)
        if task_object.new_status == self.Status.SUCCESS:
            resources = channel.sandbox.list_resources(
                task_id=task_id,
                resource_type=resource_types.TEST_ENVIRONMENT_JSON_V2.name
            )
            results_path = self.sync_resource(resources[0])
            with open(results_path) as report_file:
                loaded = json.load(report_file)
                return loaded.get("results", [])
        else:
            expected_suite = json.loads(self.ctx.get("expected_test_info", "{}"))
            if not expected_suite or not expected_suite["suite_id"]:
                raise Exception("Child task {} failed with status {}".format(task_id, task_object.new_status))

            error_type = "TIMEOUT" if task_object.new_status == self.Status.TIMEOUT else "REGULAR"
            snippet = "Task {} failed with status {}: no test results can be obtained.\nTask info: {}".format(task_id, task_object.new_status, task_object.info)
            return sdk_compat.build_error_result(self, task_id, snippet, error_type, 'FAILED')

    def _get_subtask_stats(self, task_id):
        task_object = channel.sandbox.get_task(task_id)
        if task_object.new_status == self.Status.SUCCESS:
            resources = channel.sandbox.list_resources(
                task_id=task_id,
                resource_type=sdk_resources.BUILD_STATISTICS.name
            )
            results_path = self.sync_resource(resources[0])
            return results_path

    def _is_distbuild_allowed(self, tags):
        return self._is_force_distbuild(tags) or self.owner in SEMIDIST_OWNERS

    def _is_force_distbuild(self, tags):
        return "ya:force_distbuild" in tags

    def _add_statistics_to_result(self, result, subtask_stat_path):
        cache_hit_path = os.path.join(subtask_stat_path, "cache-hit.json")
        critical_path_path = os.path.join(subtask_stat_path, "critical-path.json")
        results = result.get("results", [])
        for res in results:
            if res.get("suite", False):
                with open(cache_hit_path, 'r') as afile:
                    cache_hit_data = json.load(afile)
                    cache_hit = cache_hit_data["cache_hit"]
                with open(critical_path_path, 'r') as afile:
                    critical_path_data = json.load(afile)
                    critical_path_total_time = 0
                    for node_data in critical_path_data:
                        critical_path_total_time += node_data["elapsed"]
                res["metrics"]["sandbox.task.ya.critical_path_seconds"] = critical_path_total_time / 1000
                res["metrics"]["sandbox.task.ya.cache_hit_percent"] = cache_hit
