import collections
import copy
import hashlib
import itertools
import urlparse


def get_test_links_snippet(t, task_link):
    link_snippets = []
    default_links = {"task": [task_link]}
    for name, val in (t.get("links", default_links) or default_links).iteritems():
        link_snippets.append("{} ([[path]]{}[[rst]])".format(name, val[0]))
    return ", ".join(link_snippets)


def median(values):
    values = sorted(values)
    half, odd = divmod(len(values), 2)
    if odd:
        return values[half]
    return (values[half - 1] + values[half]) / 2.0


def merge_tests(tests, result_paths, quorum_count):
    if len(tests) == len(result_paths) == 1:
        return tests[0]

    merged = copy.deepcopy(tests[0])
    test_by_path_mapping = {t["result_path"]: t for t in tests}

    merged["links"] = {}
    for i, links in enumerate([t.get("links", {}) for t in tests]):
        for name, val in links.iteritems():
            merged["links"]["{}_{}".format(name, i + 1)] = val

    aggregated_metrics = {}
    for test_metrics in [t.get("metrics", {}) for t in tests]:
        for metric_name, metric_value in test_metrics.iteritems():
            if metric_name not in aggregated_metrics:
                aggregated_metrics[metric_name] = []
            aggregated_metrics[metric_name].append(metric_value)

    merged["metrics"] = {}
    for metric_name, metric_values in aggregated_metrics.iteritems():
        parsed = urlparse.urlparse(metric_name)
        metric_name = parsed.path
        metric_meta = urlparse.parse_qs(parsed.query)
        aggregating_funcs = {
            "avg": lambda l: sum(l) / float(len(l)),
            "min": min,
            "max": max,
            "sum": sum,
            "median": median,
        }
        aggregating_func = aggregating_funcs.get(metric_meta.get("aggr", ["avg"])[0])
        merged["metrics"][metric_name] = aggregating_func(metric_values)

    statuses = collections.defaultdict(int)
    ok_status = "{}_{}".format("OK", None)
    for t in tests:
        statuses["{}_{}".format(t["status"], t.get("error_type"))] += 1

    if len(statuses.keys()) > 1 and statuses[ok_status] < quorum_count or len(test_by_path_mapping) < quorum_count:
        merged["status"] = "FAILED"
        merged["error_type"] = "REGULAR"
    elif len(statuses.keys()) > 1:
        # otherwise tests[0] will bring the proper status to the merged test
        merged["status"] = "OK"
        if "error_type" in merged:
            del merged["error_type"]

    snippet_strings = []
    for result_path in result_paths:
        if result_path in test_by_path_mapping:
            t = test_by_path_mapping[result_path]
            if t["status"] == "OK":
                snippet_strings.append("[[good]]OK[[rst]]: {}".format(get_test_links_snippet(t, result_path)))
            elif t["status"] == "FAILED":
                snippet_strings.insert(0, "[[bad]]FAILED[[rst]]: {}".format(get_test_links_snippet(t, result_path)))
            else:
                snippet_strings.append("[[alt]]{}[[rst]]: {}".format(t["status"], get_test_links_snippet(t, result_path)))
        else:
            snippet_strings.insert(0, "[[bad]]NO_DATA[[rst]]: [[path]]{}[[rst]]".format(result_path))

    merged["rich-snippet"] = "\n".join(snippet_strings)
    return merged


def merge(tests_results, quorum_count=None):
    """
    Merges test results
    :param tests_results: test results map {<result task link>: {tests from the task}
    :param quorum_count count of results enough to form test result
    :return: merged tests
    """
    aggregated = {}
    if not quorum_count:
        quorum_count = len(tests_results)
    for result_path, tests in tests_results.iteritems():
        for record in tests:
            if record["type"] == "test":
                record["result_path"] = result_path
                test_id = record["id"]
                if test_id not in aggregated:
                    aggregated[test_id] = []
                aggregated[test_id].append(record)

    return [merge_tests(tests, tests_results.keys(), quorum_count) for tests in aggregated.itervalues()]


def imprint(data, prefix=None):
    md5 = hashlib.md5()
    for x in data:
        md5.update(x)
    res = md5.hexdigest()

    if prefix:
        return '{}-{}'.format(prefix, res)
    return res


def union(data, fun):
    return sorted(set(itertools.chain.from_iterable(map(fun, data))))


def mine_toolchain(data):
    for x in reversed(data):
        if x.get("toolchain") not in [None, "tool"]:
            return x
    return "native_build"


def native_build_merge(test_results, targets, native_build_path):
    assert len(test_results) == 1, test_results

    target_entries = [x for x in test_results.values()[-1] if x.get("path") in targets]
    build_targets = {x["path"]: x for x in target_entries if x.get("type") == "build"}
    missing = set(targets) - set(build_targets)
    config_targets = {x["path"]: x for x in target_entries if x["path"] in missing}
    target_entries = config_targets.values() + build_targets.values()

    if target_entries:
        toolchain = target_entries[-1]['toolchain']
    else:
        toolchain = mine_toolchain(test_results)

    result = {
        "id": imprint([native_build_path]),
        "name": "native_build",
        "owners": {
            "logins": union(target_entries, lambda x: x.get('owners', {}).get('logins', [])),
            "groups": union(target_entries, lambda x: x.get('owners', {}).get('groups', [])),
        },
        "path": native_build_path,
        "size": "large",
        "status": "OK",
        "suite": True,
        "toolchain": toolchain,
        "type": "test",
        "uid": imprint([x['uid'] for x in target_entries], prefix='native-test'),
    }

    broken_targets = [x for x in target_entries if x.get("status") != "OK"]

    if broken_targets:
        result.update({
            "status": "FAILED",
            "error_type": "REGULAR",
        })

        snippets = ["[[imp]]{}[[bad]]:\n{}".format(x['path'], x['rich-snippet']) for x in broken_targets]
        result["rich-snippet"] = "[[bad]]Broken targets:\n{}".format("\n".join(snippets))
        result["rich-snippet"] = result["rich-snippet"][:4000]

    missing_targets = sorted(set(targets) - set(x['path'] for x in target_entries))
    if missing_targets and result["status"] == "OK":
        result.update({
            "status": "FAILED",
            "error_type": "REGULAR",
            "rich-snippet": "[[bad]]Following targets are missing in build report:\n{}".format("\n".join(missing_targets)),
        })

    return [result]
