# -*- coding: utf-8 -*-
import os
import logging
import time
import json

from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.process import check_process_return_code
from sandbox.sandboxsdk.paths import get_logs_folder

from sandbox.projects.common import dolbilka
from sandbox.projects.common.search.components import SearchComponent


GROUP_BSCOUNT_SETTINGS = 'Bscount settings'
GROUP_BSCOUNT_LOGS = 'Bscount logs settings'
GROUP_BFG = 'BFG9000 UDP shooter settings'


class BscountResourceId(parameters.SandboxIntegerParameter):
    name = 'bscount_resource_id'
    description = 'Bscount binaries resource id'
    group = GROUP_BSCOUNT_SETTINGS
    multiline = False
    required = True
    default_value = None


class BscountDatagramPort(parameters.SandboxIntegerParameter):
    name = 'bscount_datagram_port'
    description = 'BSCOUNT_DATAGRAM_PORT'
    group = GROUP_BSCOUNT_SETTINGS
    multiline = False
    default_value = 5001


class BscountHttpPort(parameters.SandboxIntegerParameter):
    name = 'bscount_http_port'
    description = 'BSCOUNT_HTTP_PORT'
    group = GROUP_BSCOUNT_SETTINGS
    multiline = False
    default_value = 5002


class BscountHandlerThreads(parameters.SandboxIntegerParameter):
    name = 'bscount_handler_threads'
    description = 'BSCOUNT_HANDLER_THREADS'
    group = GROUP_BSCOUNT_SETTINGS
    multiline = False
    default_value = 2


class BscountShowsFrameLen(parameters.SandboxIntegerParameter):
    name = 'bscount_shows_frame_len'
    description = 'BSCOUNT_SHOWS_FRAME_LEN'
    group = GROUP_BSCOUNT_SETTINGS
    multiline = False
    default_value = 46800


class BscountStoreTime(parameters.SandboxIntegerParameter):
    name = 'bscount_store_time'
    description = 'BSCOUNT_BSCOUNT_STORE_TIME'
    group = GROUP_BSCOUNT_SETTINGS
    multiline = False
    default_value = 86400


class BscountStoreEventLog(parameters.SandboxBoolParameter):
    name = 'bscount_store_event_log'
    description = 'Store bscount event log'
    group = GROUP_BSCOUNT_LOGS
    default_value = True


class BscountStoreAccessLog(parameters.SandboxBoolParameter):
    name = 'bscount_store_access_log'
    description = 'Store bscount access log'
    group = GROUP_BSCOUNT_LOGS
    default_value = True


class UdpJsonResourceId(parameters.SandboxIntegerParameter):
    name = 'udp_json_resource_id'
    description = 'Bscount UDP json requests resource id (for BFG9000)'
    group = GROUP_BFG
    multiline = False
    required = True
    default_value = None


class BfgThreadCount(parameters.SandboxIntegerParameter):
    name = 'bfg_thread_count'
    description = 'BFG9000 UDP shooter thread count'
    group = GROUP_BFG
    multiline = False
    default_value = 1


class HttpAmmoResourceId(parameters.SandboxIntegerParameter):
    name = 'http_ammo_resource_id'
    description = ('Bscount HTTP requests resource id (for DOLBILKA). '
                   'Set ID=0 to generate HTTP ammo automatically')
    group = dolbilka.DOLBILKA_GROUP
    multiline = False
    default_value = 0


class HttpAmmoOrdersPerRequest(parameters.SandboxIntegerParameter):
    name = 'http_ammo_orders_per_request'
    description = 'A number of orders in a single HTTP request'
    group = dolbilka.DOLBILKA_GROUP
    multiline = False
    default_value = 100


class BscountDolbilkaMaximumSimultaneousRequests(dolbilka.DolbilkaMaximumSimultaneousRequests):
    # FIXME: find out the right value empiricaly!
    # FIXME: depends of CPU number
    default_value = 64


class BscountDolbilkaFixedRps(dolbilka.DolbilkaFixedRps):
    # FIXME: find out the right value empiricaly!
    # FIXME: theoretical max = 2k RPS
    default_value = 3000


class BscountDolbilkaSessionsCount(dolbilka.DolbilkaSessionsCount):
    default_value = 2


class YabsBscount(SearchComponent):
    name = 'bscount'
    START_WAIT_TIMEOUT = 300
    outputs_to_one_file = False

    def __init__(
            self,
            bscount_dir,
            bfg,
            work_dir='.',
            instance_suffix='',
            task=None,
            datagram_port=5001,
            http_port=5002,
            host='localhost',
            handler_threads=2,
            shows_frame_len=46800,
            store_time=86400,
            store_event_log=True,
            store_access_log=True,
    ):
        super(YabsBscount, self).__init__(task)

        self.instance_name = '{}{}'.format(self.name, instance_suffix)
        self.log = logging.getLogger().getChild(self.instance_name)

        # prepare paths
        self.binary_path = os.path.join(bscount_dir, 'bin')
        self.bscount_path = os.path.join(self.binary_path, 'yabs-bscount')
        self.work_dir = os.path.abspath(work_dir)

        # bscount ports
        self.datagram_port = datagram_port
        self.http_port = http_port

        # bscount logs
        self.logs_folder = get_logs_folder()

        self.stdlog = os.path.join(self.logs_folder, 'bscount_std.log')

        event_log_path = self.logs_folder if store_event_log else self.work_dir
        self.event_log = os.path.join(event_log_path, 'bscount_event.tskv')
        access_log_path = self.logs_folder if store_access_log else self.work_dir
        self.access_log = os.path.join(access_log_path, 'access.log')

        self.factor_log = os.path.join(self.work_dir, 'bscount_factor.tskv')
        self.hit_log = os.path.join(self.work_dir, 'bscount_hit.tskv')

        # another bscout settings
        self.handler_threads = handler_threads
        self.shows_frame_len = shows_frame_len
        self.store_time = store_time
        self.host = host
        self.config_hosts = '[{"fqdn":"bsgm01e.yandex.ru"}]'
        self.lock_file = 'bscount.lock'
        self.login = 'login'
        self.password = 'password'

        self.custom_env = None
        self.process = None
        self.bfg = bfg

    def _get_custom_env(self):
        current_env = os.environ
        custom_env = {
            'BSCOUNT_DATAGRAM_PORT': self.datagram_port,
            'BSCOUNT_HTTP_PORT': self.http_port,

            'BSCOUNT_EVENT_LOG': self.event_log,
            'BSCOUNT_FACTOR_LOG': self.factor_log,
            'BSCOUNT_HIT_LOG': self.hit_log,
            'BSCOUNT_ACCESS_LOG': self.access_log,

            'BSCOUNT_HOSTNAME': self.host,
            'BSCOUNT_CONFIG_HOSTS': self.config_hosts,
            'BSCOUNT_CONFIG_LOGIN': self.login,
            'BSCOUNT_CONFIG_PASSWORD': self.password,
            'BSCOUNT_LOCK_FILE': self.lock_file,
            'BSCOUNT_HANDLER_THREADS': self.handler_threads,
            'BSCOUNT_SHOWS_FRAME_LEN': self.shows_frame_len,
            'BSCOUNT_STORE_TIME': self.store_time,
        }
        custom_env = {k: str(v) for k, v in custom_env.iteritems()}
        self.log.debug('Bscount environment variables: {}'.format(json.dumps(custom_env, indent=4)))
        custom_env.update(current_env)
        if self.custom_env is None:
            self.custom_env = custom_env
        return custom_env

    def start(self):
        """
        Start process and exit
        """
        custom_env = self._get_custom_env()
        self.process = run_process(
            self.bscount_path,
            environment=custom_env,
            wait=False,
            log_prefix=self.stdlog,
        )
        self.log.info('started process with pid: %s', self.process.pid)

    def wait(self):
        """
        Wait and do pre-start checks
        """
        # it takes some miliseconds to start bscount
        time.sleep(1)

        # we need to warm up bscount's storage with UDP shooting before Dolbilka
        self.bfg.start()

        # FIXME
        # self.rss_mem_after_start = self._get_rss_memory_noexcept()

    def stop(self):
        """
        Stop process and execute checks
        """
        self.log.info('Stopping process')

        if self.process is None:
            self._raise_error("process was not started (bug in the task??)")
        if not self.is_running():
            self.log.warning("process died")

        component_output = self.process.stdout_path if self.outputs_to_one_file else self.process.stderr_path

        # FIXME
        # self.rss_mem_before_stop = self._get_rss_memory_noexcept()

        # sleep for some time to let bscount to flush all data (e.g. logs) to disk
        time.sleep(1)

        self.process.terminate()
        self.process.wait()

        self.process = None
        self.log.info('Stopped %s', self.name)

        if component_output:
            self.log.info('Verifying stderr for "%s" at "%s"', self.name, component_output)
            self.verify_stderr(component_output)
        else:
            self.log.error('Empty component output filename, stderr verification disabled for "%s"', self.name)

    def use_component(self, work_func):
        # usage_impact = {'start': self._get_usage_data_noexcept()} # FIXME: mem usage for example
        try:
            return work_func()
        finally:
            if self.is_running():
                # usage_impact['stop'] = self._get_usage_data_noexcept()    # FIXME: mem usage for example
                self.log.info("still running after use")
            else:
                self.log.info("died during use")
                self._process_post_mortem()
            # self.usage_impact_in_ctx.append(usage_impact) # FIXME: see 'usage_impact' comments


class YabsBfg(SearchComponent):
    name = 'bfg'
    START_WAIT_TIMEOUT = 300
    outputs_to_one_file = False

    def __init__(
            self,
            bfg_dir,
            udp_json_path,
            instance_suffix='',
            task=None,
            datagram_port=5001,
            target_host='127.0.0.1',
            thread_count=1,
    ):
        super(YabsBfg, self).__init__(task)

        self.instance_name = '{}{}'.format(self.name, instance_suffix)
        self.log = logging.getLogger().getChild(self.instance_name)

        # prepare paths
        self.binary_path = os.path.join(bfg_dir, 'bin')
        self.bfg_path = os.path.join(self.binary_path, 'bfg9000')

        # bfg settings
        self.datagram_port = datagram_port
        self.target_host = target_host
        self.thread_count = thread_count
        self.udp_json_path = udp_json_path

        self.process = None

    def start(self):
        """Run UDP shooting with BFG9000"""
        self.log.info(self.start.__doc__)

        cmd = [
            self.bfg_path,
            self.thread_count,
            self.target_host,
            self.datagram_port,
            self.udp_json_path,
        ]
        cmd = map(str, cmd)
        self.process = run_process(cmd, wait=True)
        check_process_return_code(self.process)


class YabsBscountPerformance(task.SandboxTask):
    input_parameters = (
        BscountResourceId,
        BscountDatagramPort,
        BscountHttpPort,
        BscountHandlerThreads,
        BscountShowsFrameLen,
        BscountStoreTime,
        BscountStoreEventLog,
        BscountStoreAccessLog,
        UdpJsonResourceId,
        BfgThreadCount,
        HttpAmmoResourceId,
        HttpAmmoOrdersPerRequest,
        dolbilka.DolbilkaRequestTimeout,
        dolbilka.DolbilkaExecutorMode,
        dolbilka.DolbilkaScheduledRps,
        BscountDolbilkaFixedRps,
        BscountDolbilkaMaximumSimultaneousRequests,
        BscountDolbilkaSessionsCount,
    )

    execution_space = 10*1024
    type = 'YABS_BSCOUNT_PERFORMANCE'
    description = 'yabs-bscount performance test'

    environment = (
        SvnEnvironment(),
    )

    @property
    def bscount_resource_id(self):
        return self.ctx[BscountResourceId.name]

    @property
    def bscount_datagram_port(self):
        return self.ctx[BscountDatagramPort.name]

    @property
    def bscount_http_port(self):
        return self.ctx[BscountHttpPort.name]

    @property
    def bscount_handler_threads(self):
        return self.ctx[BscountHandlerThreads.name]

    @property
    def bscount_shows_frame_len(self):
        return self.ctx[BscountShowsFrameLen.name]

    @property
    def bscount_store_time(self):
        return self.ctx[BscountStoreTime.name]

    @property
    def bscount_store_event_log(self):
        return self.ctx[BscountStoreEventLog.name]

    @property
    def bscount_store_access_log(self):
        return self.ctx[BscountStoreAccessLog.name]

    @property
    def udp_json_resource_id(self):
        return self.ctx[UdpJsonResourceId.name]

    @property
    def bfg_thread_count(self):
        return self.ctx[BfgThreadCount.name]

    @property
    def http_ammo_resource_id(self):
        return self.ctx[HttpAmmoResourceId.name]

    @property
    def http_ammo_orders_per_request(self):
        return self.ctx[HttpAmmoOrdersPerRequest.name]

    def on_execute(self):
        self.work_dir = os.path.abspath('.')
        self.udp_json_res_path = self.get_udp_json()
        self.http_ammo_path = self.get_http_ammo()
        bscount_res_path = self.get_bscount_path()

        bfg = YabsBfg(
            # bfg is located in bscount dir
            bfg_dir=bscount_res_path,
            udp_json_path=self.udp_json_res_path,
            task=self,
            datagram_port=self.bscount_datagram_port,
            target_host='127.0.0.1',
            thread_count=self.bfg_thread_count,
        )
        bscount = YabsBscount(
            bscount_dir=bscount_res_path,
            bfg=bfg,
            task=self,
            datagram_port=self.bscount_datagram_port,
            http_port=self.bscount_http_port,
            handler_threads=self.bscount_handler_threads,
            shows_frame_len=self.bscount_shows_frame_len,
            store_time=self.bscount_store_time,
            store_event_log=self.bscount_store_event_log,
            store_access_log=self.bscount_store_access_log,
        )

        self.run_http_shooting(bscount)

    def get_bscount_path(self):
        """
        Sync and unpack bscount if not synced&unpacked yet
        Return path to bscount
        """
        res_id = self.bscount_resource_id
        try:
            return self._bscount_path
        except AttributeError:
            self._bscount_path = self.sync_resource(res_id)
            return self._bscount_path

    def get_udp_json(self):
        """
        Sync and unpack json with UDP requests for BFG9000 if not synced&unpacked yet
        Return path to json
        """
        res_id = self.udp_json_resource_id
        try:
            return self._bscount_udp_json_path
        except AttributeError:
            self._bscount_udp_json_path = self.sync_resource(res_id)
            return self._bscount_udp_json_path

    def get_http_ammo(self):
        """
        Returns a path to Dolbilka HTTP ammo.
        """
        if self.http_ammo_resource_id:
            return self._get_http_ammo_from_resource()
        else:
            return self._generate_http_ammo_from_udp_json()

    def _get_http_ammo_from_resource(self):
        """
        Sync and unpack HTTP requests for DOLBILKA if not synced&unpacked yet
        Return path to json
        """
        res_id = self.http_ammo_resource_id
        try:
            return self._http_ammo_path
        except AttributeError:
            self._http_ammo_path = self.sync_resource(res_id)
            return self._http_ammo_path

    def _generate_http_ammo_from_udp_json(self):
        """Generate Dolbilka HTTP ammo using BFG9000's UDP json"""
        logging.info(self._generate_http_ammo_from_udp_json.__doc__)

        bscount_http_ammo_path = os.path.join(self.work_dir, 'http-requests.txt')
        current_time = int(time.time())

        with open(self.udp_json_res_path, 'r') as udp_requests_json_file, \
                open(bscount_http_ammo_path, 'w') as dolbilka_http_ammo_file:
            udp_requests_json = json.load(udp_requests_json_file)

            http_request_body_start = (
                '2\n'
                '{counter}\t{current_time}'.format(
                    counter=self.http_ammo_orders_per_request,
                    current_time=current_time,
                )
            )
            http_request_body = ''
            for counter, udp_request in enumerate(udp_requests_json, 1):
                http_request_body_part = '{order_id}\t{start_time}\t0'.format(
                    order_id=udp_request['OrderID'],
                    start_time=(current_time - 1000),
                )
                if not http_request_body:
                    http_request_body = http_request_body_start
                http_request_body += '\n' + http_request_body_part

                if not (counter % self.http_ammo_orders_per_request):
                    http_request_headers = (
                        'POST /count/ HTTP/1.1\n'
                        'Host: bscount.yandex.ru\n'
                        'Content-Length: {content_len}\n'
                        'Content-Type: text/plain\n'
                        '\n'.format(
                            content_len=len(http_request_body),
                        )
                    )
                    http_request = http_request_headers + http_request_body + '\n'
                    http_request_body = ''

                    dolbilka_header = '{http_request_len} 0\n'.format(
                        http_request_len=len(http_request),
                    )
                    dolbilka_http_ammo_file.write(dolbilka_header + http_request)

        return bscount_http_ammo_path

    def run_http_shooting(self, bscount):
        """Run HTTP shooting with Dolbilka"""
        logging.info(self.run_http_shooting.__doc__)

        d_planner = dolbilka.DolbilkaPlanner()
        d_executor = dolbilka.DolbilkaExecutor()

        bscount_d_plan = d_planner.create_plan(
            self.http_ammo_path,
            loader_type='phantom',
            # avoid subtle v4/v6 issues
            host='127.0.0.1',
            port=self.bscount_http_port,
        )

        results = d_executor.run_sessions(bscount_d_plan, bscount, run_once=True)
        dolbilka.DolbilkaPlanner.fill_rps_ctx(results, self.ctx)


__Task__ = YabsBscountPerformance
