# -*- coding: utf-8 -*-

import logging
import re
import sandbox.sandboxsdk.task as sdk_task
import sandbox.common.types.client as ctc
from sandbox.sdk2 import yav
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import parameters
from multiprocessing import Pool, Manager
from itertools import izip, repeat
import time
from sandbox.projects.common.yabs.juggler import (
    ServiceStatus, Juggler, STATUS_OK
)

SECRET_ID = "sec-01dc7bcaz5cfmtbbft03p4107b"
KEY_ID = "yt_token"

METRIC_PREFIX = "logs"
HOST_VERSION = 'precise'
ITER_DURATION = 60  # seconds
LOOP_DURATION = 15 * 60
LOCK_PATH = "//tmp/YabsCollectorsMonLock"


class Cluster(parameters.SandboxBoolGroupParameter):
    name = 'Cluster'
    description = 'Выберите кластеры которые будет мониторить таск'
    choices = (
        ('zeno.yt.yandex.net', 'zeno.yt.yandex.net'),
        ('seneca-sas.yt.yandex.net', 'seneca-sas.yt.yandex.net'),
        ('seneca-man.yt.yandex.net', 'seneca-man.yt.yandex.net'),
        ('seneca-vla.yt.yandex.net', 'seneca-vla.yt.yandex.net'),
        ('arnold.yt.yandex.net', 'arnold.yt.yandex.net'),
        ('hahn.yt.yandex.net', 'hahn.yt.yandex.net'),
        ('freud.yt.yandex.net', 'freud.yt.yandex.net'),
        ('bohr.yt.yandex.net', 'bohr.yt.yandex.net'),
        ('landau.yt.yandex.net', 'landau.yt.yandex.net'),
    )
    default_value = 'zeno.yt.yandex.net'


class WindowSize(parameters.SandboxIntegerParameter):
    name = 'WindowSize'
    description = 'Размер окна в котором будет мониториться load'
    default_value = 15 * 60  # 15 минут


def get_lock_count(ytc, path):
    locks = ytc.get_attribute(path, "locks")
    return sum(1 for lk in locks if lk["mode"] == "exclusive")


def _expand_config(config_dict, all_clusters):
    overall_checks = config_dict.pop("OVERALL")
    for cluster in all_clusters:
        cluster_conf = config_dict.setdefault(cluster, {})
        cluster_conf.update({k: v for k, v in overall_checks.iteritems() if k not in config_dict[cluster]})
        for check in cluster_conf:
            if check in overall_checks:
                cluster_conf[check].update({k: v for k, v in overall_checks[check].iteritems() if k not in cluster_conf[check]})
    return config_dict


def send_metrics(cluster_url, header, data, metric_name, default_prefix, sender, time_prefix):
    if '.' not in cluster_url:
        cluster_url += '.yt.yandex.net'
    if header:
        for name in header:
            sender.append(time_prefix, cluster_url, default_prefix, name, metric_name, data.get(name, 0))
    else:
        for key in data:
            if isinstance(key, tuple):
                name = key[0]
            else:
                name = key
            sender.append(time_prefix, cluster_url,  metric_name, default_prefix, name, data.get(key, 0))


def process_juggler(cluster, service, data, check_function):
    status, descr = check_function(data)
    if '.' in cluster:
        cluster = cluster.split('.')[0]
    check_name = 'sbyt-{}.yt.yandex.net'.format(cluster)
    logging.info("Sending: {} {} {} {}".format(check_name, service, status, descr))
    if status == STATUS_OK:
        Juggler().send([ServiceStatus(check_name, service, STATUS_OK)])
    else:
        Juggler().send([ServiceStatus(check_name, service, status, descr)])


def _prepare_data_for_juggler(data, expanded_config):
    def check_is_excluded(check_name):
        for exclude_pattern in expanded_config[check]["exclude"]:
            try:
                if re.match(exclude_pattern, check_name) is not None:
                    return True
            except Exception as error:
                raise Exception("Failed to match pattern: {} with check_name: {}. Error: {}".format(exclude_pattern, check_name, error.message))
        return False

    checks_data = {k: v for k, v in data.iteritems() if k in expanded_config}
    for check in checks_data:
        checks_data[check] = {k: v for k, v in checks_data[check].iteritems() if not check_is_excluded(k)}
    return checks_data


def _prepare_counted_data(extr, window_size):
    log_types = extr.get_logtypes_collectors_map()

    def _group_collector_data(data_dict):
        result = {}
        for log_type, collectors in log_types.iteritems():
            for collector in collectors:
                if collector in data_dict:
                    result[log_type + "." + collector] = data_dict[collector]
        return result

    logging.info("Total log types: {}".format(len(log_types)))
    logtype_max_time, collector_max_time = extr.get_logtypes_collectors_proc_time()
    logtype_max_time.update(_group_collector_data(collector_max_time))
    logtype_load, collector_load = extr.get_logtypes_collectors_rough_load(window_size)
    logtype_load.update(_group_collector_data(collector_load))

    header = []
    for log_name, collectors in log_types.iteritems():
        header.append(log_name)
        for collector in collectors:
            header.append(log_name + "." + collector)

    return {
        "count": extr.get_logs_count(),
        "max_processing_time": logtype_max_time,
        "load": logtype_load,
        "rows": extr.get_logtype_row_count(),
        "log_delay": extr.get_logtypes_reader_lag(),
        "log_age": extr.get_logtypes_oldest_log_age(),
        "dict_lag": extr.get_replicated_dicts_lag(extr.DICTS_PATH),
        "dict_preprod_lag": extr.get_replicated_dicts_lag(extr.DICTS_PREPROD_PATH),
        "stat_replica_lag": extr.get_replicated_tables_lag(),
    }, header


def _get_target_pos(expression):
    last_n = -1
    for n, t in enumerate(expression.split('.')):
        if t == "*":
            last_n = n
    return last_n


def _prepare_graphite_data(graph, expanded_config, cluster):
    result = {}
    if '.' in cluster:
        cluster = cluster.split('.')[0]
    for check, content in expanded_config.iteritems():
        if "graphite_metric" in content:
            result[check] = graph.get_graphite_points_with_interp(
                content["graphite_metric"].format(cluster=cluster),
                content.get("target_position", _get_target_pos(content["graphite_metric"]))
            )
    return result


def process_single_cluster(cluster_url, token, extr_class, graph_class, window_size, queue):
    logging.info("Processing cluster {}".format(cluster_url))
    try:
        extr = extr_class(cluster_url, token)
        graph = graph_class()
        logging.info("Preparing data for graphite")
        data_dict, header = _prepare_counted_data(extr, window_size)

        logging.info("Preparing logs data for graphite")
        log_dict = {k: v for k, v in data_dict.iteritems() if k not in ("dict_lag", "dict_preprod_lag", "stat_replica_lag")}
        dict_dict = {k: v for k, v in data_dict.iteritems() if k in ("dict_lag", "dict_preprod_lag", "stat_replica_lag")}
        map(lambda x: send_metrics(cluster_url, header, x[1], x[0], METRIC_PREFIX, graph, graph_class.ONE_MIN), log_dict.items())
        logging.info("Preparing dicts data for graphite")
        map(lambda x: send_metrics(cluster_url, [], x[1], x[0], "", graph, graph_class.ONE_MIN), dict_dict.items())
        logging.info("Sending data to graphite")
        graph.push()
        queue.put((cluster_url, data_dict))

    except Exception:
        logging.exception("Failed processing cluster {}".format(cluster_url))


def juggler_single_cluster(cluster_url, data_dict, graph_class, expanded_config):
    graph = graph_class()
    cluster_config = expanded_config[cluster_url]
    logging.info("Preparing data for checks")
    data_dict.update(_prepare_graphite_data(graph, cluster_config, cluster_url))
    checks_data = _prepare_data_for_juggler(data_dict, cluster_config)
    logging.info("Sending to juggler")
    map(lambda x: process_juggler(cluster_url, x[0], x[1], cluster_config[x[0]]["check"]), checks_data.items())


def process_single_cluster_star(args):
    process_single_cluster(*args)


class Task(sdk_task.SandboxTask):
    type = "YABS_COLLECTORS_MON"
    environment = (
        environments.PipEnvironment("yandex-yt", use_wheel=True),
    )
    # На некоторых машинах нет wheel с Yt,
    # поэтому фильтр
    client_tags = ctc.Tag.LINUX_PRECISE
    cores = 1
    required_ram = 4096
    execution_space = 4096

    input_parameters = (
        Cluster,
        WindowSize,
    )

    def on_execute(self):
        import yt.wrapper as yt
        from . import worker
        from . import checks_config
        start_time = time.time()
        logging.basicConfig(level=logging.INFO)
        clusters = self.ctx.get("Cluster").split(' ')
        window_size = int(self.ctx.get("WindowSize"))

        token = yav.Secret(SECRET_ID).data()[KEY_ID]
        ytc = yt.YtClient(config={"token": token, "proxy": {"url": "locke"}})
        expanded_config = _expand_config(checks_config.CHECKS_CONFIG, clusters)
        ytc.create("int64_node", LOCK_PATH, ignore_existing=True, recursive=True)
        if get_lock_count(ytc, LOCK_PATH) > 1:
            return
        p = Pool(10)
        with ytc.Transaction():
            ytc.lock(LOCK_PATH, mode="exclusive", waitable=True, wait_for=600000)
            m = Manager()
            while time.time() - start_time < LOOP_DURATION and get_lock_count(ytc, LOCK_PATH) == 1:
                iter_start = time.time()
                q = m.Queue()
                p.map(
                    process_single_cluster_star,
                    izip(
                        clusters,
                        repeat(token),
                        repeat(worker.LogsMonitorExtractor),
                        repeat(worker.GraphiteWorker),
                        repeat(window_size),
                        repeat(q)
                    )
                )

                while not q.empty():
                    cluster, data = q.get()
                    logging.info("Jugglering cluster {}".format(cluster))
                    juggler_single_cluster(cluster, data, worker.GraphiteWorker, expanded_config)
                sleep_duration = iter_start + ITER_DURATION - time.time()
                if sleep_duration > 0:
                    time.sleep(sleep_duration)


__Task__ = Task
