# -*- coding: utf-8 -*-

import re
import time

from sandbox.projects.common.yabs.juggler import STATUS_OK, STATUS_WARN, STATUS_CRIT


def match_threshold(name, default_thresholds, custom_thresholds):
    if isinstance(name, tuple):
        name = name[0]

    thresholds = default_thresholds
    if custom_thresholds is not None:
        longest_pattern = 0
        for pattern, pattern_thresholds in custom_thresholds.iteritems():
            if len(pattern) > longest_pattern and re.match(pattern, name):
                longest_pattern = len(pattern)
                thresholds = pattern_thresholds

    if "night" not in thresholds:
        thresholds["night"] = 2 ** 30

    return thresholds


def is_night():
    h = int(time.strftime("%H"))
    if h >= 22 or h <= 9:
        return True

    return False


def more_than_limit_check(points_dict, default_thresholds, desc, custom_thresholds=None):
    result = STATUS_OK
    description = []
    for name, value in points_dict.iteritems():
        thresholds = match_threshold(name, default_thresholds, custom_thresholds)
        threshold_crit = thresholds["night"] if is_night() else thresholds["crit"]
        threshold_warn = thresholds["warn"]

        if value > threshold_crit:
            result = STATUS_CRIT
            if isinstance(name, tuple):
                description.append((value, desc.format(name[0], name[1], value, threshold_crit)))
            else:
                description.append((value, desc.format(name, value, threshold_crit)))
        elif value > threshold_warn:
            if result != STATUS_CRIT:
                result = STATUS_WARN
            if isinstance(name, tuple):
                description.append((value, desc.format(name[0], name[1], value, threshold_warn)))
            else:
                description.append((value, desc.format(name, value, threshold_warn)))

    sorted_descriptions = [d[1] for d in sorted(description, key=lambda x: x[0], reverse=True)]
    return result, " | ".join(sorted_descriptions)


LOG_DELAY_CONFIG = {
    "offline_joined-qtail-buffer-log": {"crit": 3600 * 20, "warn": 3600 * 6},
    "offline_bs-chevent-cooked-log_1d": {"crit": 3600 * 24, "warn": 3600 * 15},
    "offline_bs.*_1h": {"crit": 3600 * 20, "warn": 3600 * 6},
    "broad-phrase-dynamic": {"crit": 3600 * 40, "warn": 3600 * 25},
    "bs-chevent-log": {"night": 3600 * 2, "crit": 3600, "warn": 1800},
    "bs-ally-services-billed-checked-log": {"night": 3600 * 12, "crit": 3600 * 4, "warn": 3600 * 3},
    "zen-billed-log": {"night": 3600 * 12, "crit": 3600 * 4, "warn": 3600 * 3},
    "bs-dsp-checked-log": {"night": 3600 * 2, "crit": 4200, "warn": 2700},
    "bs-undodsp-log": {"night": 3600 * 2, "crit": 5400, "warn": 4200},
    "visit-v2-ads-log": {"night": 3600 * 3, "crit": 3600 * 3, "warn": 3600},
    "bs-dsp-creative-moderation-response-log": {"crit": 3600 * 3, "warn": 3600 * 2},
    "bs-proto-mismatchedpatterns-log": {"crit": 3600 * 3, "warn": 3600 * 2},
    "bs-proto-dspcreative-log": {"crit": 3600 * 3, "warn": 3600 * 2},
    "bs-proto-candidates-log": {"crit": 3600 * 2, "warn": 3600 * 1},
}

LOG_AGE_CONFIG = {
    "offline_joined-qtail-buffer-log": {"crit": 3600 * 1, "warn": 3600 / 2},
    "offline_bs-chevent-cooked-log_1d": {"crit": 3600 * 48, "warn": 86400 + 3600 * 10},
    "offline_bs.*_1h": {"crit": 3600 * 20, "warn": 3600 * 7},
    "broad-phrase-dynamic": {"crit": 3600 * 40, "warn": 3600 * 25},
    "bs-chevent-log": {"night": 3600 * 6, "crit": 3600 * 4, "warn": 9000},
    "visit-v2-ads-log": {"night": 3600 * 3, "crit": 3600 * 3, "warn": 3600},
    "bs-dsp-creative-moderation-response-log": {"crit": 3600 * 6, "warn": 3600 * 3},
    "bs-proto-mismatchedpatterns-log": {"crit": 3600 * 6, "warn": 3600 * 3},
    "bs-proto-dspcreative-log": {"crit": 3600 * 6, "warn": 3600 * 3},
}

LBYT_LAG_CONFIG = {
    "chevent": {"night": 3600, "crit": 1800, "warn": 900},
}

STATISTIC_AGE_CONFIG = {
    "(BannerPhrase|BannerContext|PageAlg)": {"night": 3600 * 2, "crit": 3600, "warn": 2400},
    "BillableEventLogGenerator.*": {"night": 3600 * 4, "crit": 3600 * 3, "warn": 3600 * 2},
    "Advm.*Chevent": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    "MdbMasterRep.*": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    "MdbMasterReportStatVisitV2|DirectGridVisitV2|DirectGridGoalsVisitV2": {"night": 3600 * 4, "crit": 3600 * 3, "warn": 3600 * 2},
    "AuditBids|ChEventArchive": {"crit": 3600 * 6, "warn": 9000},
    "ExperimentStat": {"crit": 3600 * 6, "warn": 3600 * 4},
    "QueryTail": {"crit": 3600 * 21, "warn": 3600 * 18},
    ".*DspChecked.*": {"crit": 3600 * 2, "warn": 3600},
    ".*OutdoorTracking": {"crit": 3600 * 48 * 365, "warn": 3600 * 48 * 365},
    "UnrolledShows": {"crit": 3600 * 24 * 3, "warn": 3600 * 24},
    ".*VisitV2": {"crit": 10800, "warn": 7200},
    "Advm.*Visit": {"night": 10800, "crit": 10800, "warn": 6500},
    "DSPCreativeModerated": {"crit": 3600 * 6, "warn": 3600 * 3},
    "BrandSafety*": {"crit": 3600 * 6, "warn": 3600 * 3},
    "SendDSPCreativesToModeration": {"crit": 3600 * 6, "warn": 3600 * 3},
    "AdfoxEventCollectorUndo": {"crit": 3600 * 8760 * 30, "warn": 3600 * 8760 * 30},
    ".*UndoChTracking.*": {"crit": 3600 * 12, "warn": 3600 * 8},
}

DIRECT_STATISTIC_AGE_CONFIG = {
    "DirectGridGoalsVisitV2|DirectGridGoalsVisitV2|DirectPhraseGoalsVisitV2|DirectPhraseVisitV2|OrderStatDayVisitV2": {"night": 3600 * 4, "crit": 3600 * 3, "warn": 3600 * 2},
    "DirectGridUndoChevent|DirectPhraseUndoChevent|OrderStatFraudUndoChevent|OrderStatDayUndoChevent|OrderStatUndoChevent|OrderStatDeleteUndoChEvent": {"crit": 3600 * 6, "warn": 3600 * 4},
    "DirectGrid.*": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    "DirectPhrase.*": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    "OrderStatDay.*": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    "OrderStatFraud.*": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    "OrderStatChevent": {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600},
    ".*UndoChTracking.*": {"night": 3600 * 24, "crit": 3600 * 12, "warn": 3600 * 8},
}

CHECKS_CONFIG = {
    "OVERALL": {
        "log_delay": {
            "check": lambda x: more_than_limit_check(x, {"crit": 3600, "warn": 1800}, "Log delay for [{}] is {} > {} limit", LOG_DELAY_CONFIG),
            "exclude": ["bs-undo.*-prestable-log", "offline_bs-chhit-log_1d"],
        },
        "log_age": {
            "check": lambda x: more_than_limit_check(x, {"crit": 3600 * 2, "warn": 3600}, "The oldest log age for [{}] is {} > {} limit", LOG_AGE_CONFIG),
            "exclude": ["offline_bs-chhit-log_1d"],
        },
        "lbyt_lag": {
            "check": lambda x: more_than_limit_check(x, {"crit": 5400, "warn": 1800}, "LBYT lag (oldest task) for [{}] is {} > {} limit", LBYT_LAG_CONFIG),
            "exclude": ["xurma-.*"],
            "graphite_metric": "five_sec.{cluster}.lbyt.*.lag",
        },
        "lbyt_errors": {
            "check": lambda x: more_than_limit_check(x, {"crit": 10000, "warn": 1000}, "LBYT parsing errors count for [{}] is {} > {} limit"),
            "exclude": ["xurma-.*"],
            "graphite_metric": "five_sec.{cluster}.lbyt.*.recent_parsing_errors",
        },
        "offsets": {
            "check": lambda x: more_than_limit_check(x, {"crit": 30, "warn": 15}, "Offsets available for [{}] is {} > {} than were read for LbYt reader"),
            "graphite_metric": "five_sec.bs.lbyt.*-{cluster}.*.relative_diff",
            "exclude": [],
        },
        "statistics_age": {
            "check": lambda x: more_than_limit_check(x, {"crit": 7200, "warn": 3000}, "Statistics age for [{}] is {} > {} limit", STATISTIC_AGE_CONFIG),
            "graphite_metric": "one_min.{cluster}.*.*-log.delay",
            "exclude": [".*UndoDsp", ".*UndoCh[Tt]racking", ".*UndoCh[eE]vent", ".*MobileInstall",
                        ".*BsAllyServicesBilledLog", ".*ZenCheckedBilledLog", ".*AllyCheckedBilled", ".*AllyServicesBilledChecked"],
        },
        "dict_lag": {
            "check": lambda x: more_than_limit_check(x, {"night": 900, "crit": 600, "warn": 300}, "Dict lag for [{}] is {} > {} limit"),
            "exclude": [],
        },
        "dict_preprod_lag": {
            "check": lambda x: more_than_limit_check(x, {"crit": 1800, "warn": 1200}, "Dict lag for [{}] is {} > {} limit"),
            "exclude": [],
        },
        "mdb_master_report_statistics_age": {
            "check": lambda x: more_than_limit_check(x, {"night": 7200, "crit": 7200, "warn": 3000}, "Statistics age for [{}] is {} > {} limit"),
            "graphite_metric": "one_min.{cluster}.MdbMasterReportStatChevent.*.delay",
            "target_position": 2,  # number of token in graphite_metric ({0}.{1}.{2}.{3} and so on), used as a name to group metrics by
            "exclude": [],
        },
        "direct_statistics_age": {
            "check": lambda x: more_than_limit_check(x, {"night": 3600 * 3, "crit": 3600 * 2, "warn": 3600}, "Statistics age for [{}] is {} > {} limit", DIRECT_STATISTIC_AGE_CONFIG),
            "graphite_metric": "one_min.{cluster}.{{DirectGrid*,DirectPhrase*,OrderStatDay*,OrderStatFraud*,OrderStatChevent,OrderStatUndoChevent,OrderStatDeleteUndoChEvent}}.*-log.delay",
            "target_position": 2,  # number of token in graphite_metric ({0}.{1}.{2}.{3} and so on), used as a name to group metrics by
            "exclude": [".*UndoDsp", ".*MobileInstall", ".*AllyServicesBilledChecked"],
        },
    },
    "arnold.yt.yandex.net": {
        "stat_replica_lag": {
            "check": lambda x: more_than_limit_check(x, {"crit": 7200, "warn": 3600}, "Stat replica lag for [{} -> {}] is {} > {} limit"),
            "exclude": [],
        },
        "adfox_statistics_age": {
            "check": lambda x: more_than_limit_check(x, {"night": 3600, "crit": 2700, "warn": 1800},
                                                     "Statistics age {} for [AdfoxEventCollector] is {} > {} limit"),
            "graphite_metric": "one_min.arnold.AdfoxEventCollector.adfox-event-log.delay",
            "exclude": [],
        },
        "adfox_statistics_age_spi": {
            "check": lambda x: more_than_limit_check(x, {"night": 10 * 3600, "crit": 6 * 3600, "warn": 4 * 3600},
                                                     "Statistics age {} for [AdfoxEventCollector] is {} > {} limit"),
            "graphite_metric": "one_min.arnold.AdfoxEventCollector.adfox-event-log.delay",
            "exclude": [],
        },
    },
    "hahn.yt.yandex.net": {
        "stat_replica_lag": {
            "check": lambda x: more_than_limit_check(x, {"crit": 7200, "warn": 3600}, "Stat replica lag for [{} -> {}] is {} > {} limit"),
            "exclude": [],
        },
        "adfox_statistics_age": {
            "check": lambda x: more_than_limit_check(x, {"night": 3600, "crit": 2700, "warn": 1800},
                                                     "Statistics age {} for [AdfoxEventCollector] is {} > {} limit"),
            "graphite_metric": "one_min.hahn.AdfoxEventCollector.adfox-event-log.delay",
            "exclude": [],
        },
        "adfox_statistics_age_spi": {
            "check": lambda x: more_than_limit_check(x, {"night": 10 * 3600, "crit": 6 * 3600, "warn": 4 * 3600},
                                                     "Statistics age {} for [AdfoxEventCollector] is {} > {} limit"),
            "graphite_metric": "one_min.hahn.AdfoxEventCollector.adfox-event-log.delay",
            "exclude": [],
        },
    },
    "freud.yt.yandex.net": {
        "stat_replica_lag": {
            "check": lambda x: more_than_limit_check(x, {"crit": 7200, "warn": 3600}, "Stat replica lag for [{} -> {}] is {} > {} limit"),
            "exclude": [],
        }
    },
}
