# -*- coding: utf-8 -*-
# for creating pbuilder image:
# 1) pbuilder --crease trusty.tgz --distribution trusty
# 2) sudo pbuilder --login --save-after-login --basetgz trusty.tgz
# 3) remove any extra block devices: https://st.yandex-team.ru/SANDBOX-2958#56d84db7e4b0eac497c94f7a
# 4) ya upload trusty.tgz -T YABS_PBUILDER_IMAGE --do-not-remove -A release=trusty

import logging
import re
import os
import shutil
import zipfile
import json
import platform
import time

from shutil import copy
from urllib2 import urlopen

from sandbox.common import rest
from sandbox.common import errors
from sandbox.common.types.client import Tag

from sandbox.sandboxsdk import environments

from sandbox.sandboxsdk.ssh import Key
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxSubprocessError
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.gnupg import GpgKey
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.yabs.bamboo import bamboo_build_notify
from sandbox.projects.common.arcadia import sdk as arcadiasdk

SUPPORTED_UBUNTU_VERSIONS = ['precise', 'trusty']
DEFAULT_UBUNTU_VERSION = 'precise'
DEFAULT_DISTRIBUTION = 'unstable'
DEFAULT_TESTING_DISTRIBUTION = 'testing'
WAIT_REPO_TIMEOUT = 600

DUPLOAD_CONF = {
    'yabs-precise': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/yabs-precise/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'yabs-trusty': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/yabs-trusty/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },

    'yabs-antifraud': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/yabs-antifraud/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'yabs-antifraud-precise': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/yabs-antifraud-precise/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'yabs-antifraud-trusty': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/yabs-antifraud-trusty/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },

    'qabs-bamboo-precise': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/qabs-bamboo-precise/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'qabs-bamboo-trusty': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/qabs-bamboo-trusty/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'afpb-bamboo-precise': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/afpb-bamboo-precise/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'afpb-bamboo-trusty': {
        'fqdn': 'bsdist.yandex.net',
        'method': 'scpb',
        'incoming': '/repo/afpb-bamboo-trusty/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },

    'yandex-precise': {
        'fqdn': 'yandex-precise.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/yandex-precise/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'yandex-trusty': {
        'fqdn': 'yandex-trusty.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/yandex-trusty/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },

    'system': {
        'fqdn': 'system.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/system/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
    'common': {
        'fqdn': 'common.dupload.dist.yandex.ru',
        'method': 'scpb',
        'incoming': '/repo/common/mini-dinstall/incoming/',
        'dinstall_runs': 0,
        'login': 'yabs-commiter',
        'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
    },
}


class ArcadiaPath(SandboxArcadiaUrlParameter):
    name = 'ArcadiaPath'
    description = 'ArcadiaPath. Path in arcadia.yandex.ru, included debian folder. Type `svn info` and copy-paste URL'
    required = True


class ArcadiaRevision(SandboxIntegerParameter):
    name = 'ArcadiaRevision'
    description = 'Revision. Check out repo in this revision (by default is HEAD)'


class ArcadiaPatch(SandboxStringParameter):
    name = 'ArcadiaPatch'
    description = 'Apply patch before debuild (diff should be taken from trunk root), use can use http-link to RB'
    default_value = ''
    multiline = True


class PackageVersion(SandboxStringParameter):
    name = 'PackageVersion'
    description = 'Version. Default is {svn-revision} or {svn-revision}_{sandbox taskid}'


class Repositories(SandboxStringParameter):
    name = 'Repositories'
    description = 'Repositories. By default use .build defined repositories OR yabs-precise/unstable,yabs-trusty/unstable'


class DryRun(SandboxBoolParameter):
    name = 'DryRun'
    description = 'Do not upload to repositories'
    default_value = False


class TestingBuild(SandboxBoolParameter):
    name = 'TestingBuild'
    description = 'Upload to qabs- repositories, or defined in testing-repositories .build option'
    default_value = False


class UseTestingSources(SandboxBoolParameter):
    name = 'UseTestingSources'
    description = 'Add testing and unstable repositories during build'
    default_value = False


class RunBambooTests(SandboxBoolParameter):
    name = 'RunBambooTests'
    description = 'Run bamboo tests'
    default_value = True


class TestIssue(SandboxStringParameter):
    name = 'TestIssue'
    description = 'Startrek issue for Bamboo tests results'
    required = False


class CheckAlreadyUploaded(SandboxBoolParameter):
    name = 'CheckAlreadyUploaded'
    description = 'Check, that packages with that version were already uploaded to repo'
    default_value = True


class YabsDebuilderDev(SandboxTask):
    privileged = True
    type = 'YABS_DEBUILDER_DEV'
    client_tags = Tag.YABS & (Tag.LINUX_PRECISE | Tag.LINUX_TRUSTY)
    environment = (
        environments.SvnEnvironment(),
        environments.SvnEnvironment(platform=platform.platform())
    )
    input_parameters = (
        ArcadiaPath,
        ArcadiaRevision,
        ArcadiaPatch,
        PackageVersion,
        UseTestingSources,
        DryRun,
        Repositories,
        TestingBuild,
        RunBambooTests,
        TestIssue,
        CheckAlreadyUploaded
    )

    def _is_empty_autocheck_build(self):
        # FIXME do we really need this??
        """
        When called from autocheck, Testenv runs two builds.
        In some cases (when??) one of them has empty context.
        Return True if this is the build with empty context.
        """
        if 'AUTOCHECK_FROM_REVIEW_BOARD' in self.ctx.get('__GSID', '') and not\
                (self.ctx.get('ArcadiaPath') and self.ctx.get('ArcadiaRevision')):
            self.set_info("Autocheck build with empty ArcadiaPath or empty ArcadiaRevision")
            return True
        return False

    def on_prepare(self):
        if self._is_empty_autocheck_build():
            return 1

        if self.ctx.get('arcadia_patch'):
            # zipatch from testenv (zipatch:http://proxy.sandbox.yandex-team.ru/NNNN)
            self.ctx['ArcadiaPatch'] = self.ctx.get('arcadia_patch')
            self.ctx['ArcadiaPath'] = self._get_arcadia_path_from_zipatch(re.compile(r':').split(self.ctx.get('arcadia_patch'), 1)[-1])

        parsed_url = Arcadia.parse_url(self.ctx.get('ArcadiaPath'))
        if self.ctx.get('ArcadiaRevision'):
            parsed_url = parsed_url._replace(revision=self.ctx.get('ArcadiaRevision'))
        if parsed_url.trunk:
            self.src_dir_root = Arcadia.get_arcadia_src_dir(Arcadia.trunk_url())
            self.src_dir = parsed_url.subpath
        elif parsed_url.branch:
            # branch was made by cp all trunk
            # ya clone && ya make --checkout must be in on_execute
            pass
        else:
            self.src_dir_root = Arcadia.get_arcadia_src_dir(self.ctx.get('ArcadiaPath'))
            self.src_dir = None

        logging.debug("checking out yabs-debuilder")
        self.yabs_debuilder = Arcadia.get_arcadia_src_dir(Arcadia.trunk_url('yabs/utils/yabs-debuilder'))

    def on_execute(self):
        if self._is_empty_autocheck_build():
            return 1

        self._update_environ()
        build_options = self._get_build_options()

        parsed_url = Arcadia.parse_url(self.ctx.get('ArcadiaPath'))
        if parsed_url.branch:
            self.src_dir_root = arcadiasdk.do_clone(self.ctx.get('ArcadiaPath'), self)
            self.src_dir = os.path.join(self.src_dir_root, parsed_url.subpath)
            ya = Arcadia.export('svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya', 'ya')
            logging.debug(os.environ)
            run_process(
                [ya, 'make'] +
                build_options.get('ya_make_options', '-j0').split(" ") +
                ['--checkout', parsed_url.subpath],
                work_dir=self.src_dir_root, log_prefix='ya_make_checkout'
            )
        else:
            self._overlay_src_dir()

        if self.ctx.get('ArcadiaPatch'):
            matched_results = re.search('rb.yandex-team.ru/.*/(\d+)', self.ctx.get('ArcadiaPatch'))
            if matched_results:
                # link to review board
                Arcadia.apply_patch(self.src_dir_root, 'rb:' + matched_results.group(1), self.abs_path())
            else:
                # diff text or zipatch
                Arcadia.apply_patch(self.src_dir_root, self.ctx.get('ArcadiaPatch') + '\n', self.abs_path())
            logging.debug('patch completed')

        run_process("sudo apt-get update && sudo apt-get install pbuilder -q0 --assume-yes --force-yes -o Dpkg::Options::='--force-confdef'", shell=True, log_prefix='install_pbuilder')
        self._build(self.src_dir_root, self.src_dir, build_options)

    def _overlay_src_dir(self):
        root = self.abs_path("src_dir")
        if os.path.exists(root):
            shutil.rmtree(root)
        delta = os.path.join(root, "delta")
        work = os.path.join(root, "work")
        merged = os.path.join(root, "merged")
        map(os.mkdir, (root, delta, work, merged))
        cmd = [
            "/bin/mount", "-t", "overlay", "overlay",
            "-olowerdir={0},upperdir={1},workdir={2}".format(self.src_dir_root, delta, work),
            merged
        ]
        logging.info("Overlaying sources at %r to %r by command %r", self.src_dir_root, merged, cmd)
        run_process(cmd, log_prefix="overlay_src_dir")
        self.src_dir_root = merged
        self.src_dir = self.src_dir_root if not self.src_dir else os.path.join(self.src_dir_root, self.src_dir)
        logging.debug("src_dir_root: %s", self.src_dir_root)
        logging.debug("src_dir: %s", self.src_dir)

    def _update_environ(self):
        # with that environs pdebuild failed with strage error "dpkg-deb: error: failed to make temporary file (control member): No such file or directory"
        if 'TEMP' in os.environ:
            del os.environ['TEMP']
        if 'TMPDIR' in os.environ:
            del os.environ['TMPDIR']
        if 'TMP' in os.environ:
            del os.environ['TMP']

        # it used for debsign
        os.environ['DEBFULLNAME'] = 'YABS dev. team'
        os.environ['EMAIL'] = 'yabs-dev@yandex-team.ru'

        # used by `ya` tool
        os.environ['YA_CACHE_DIR'] = self.abs_path('tmp', 'ya')
        os.environ['YA_CACHE_DIR_TOOLS'] = self.abs_path('tmp', 'ya_tools')

        # it used for arcadia
        os.environ['SVN_SSH'] = 'ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null'

        logging.debug(os.environ)

    def _get_arcadia_path_from_zipatch(self, zipatch):
        responce = urlopen(zipatch)
        tmp_file = os.path.join(self.abs_path(), "tmp_zipatch")
        f = open(tmp_file, "wb")
        f.write(responce.read())
        f.close()

        actions = json.loads(zipfile.ZipFile(tmp_file).read("actions.json"))
        for action in actions:
            # NOTE: works only with trunk
            path = Arcadia.trunk_url() + '/' + action['path']
            while len(path) > len(Arcadia.trunk_url()):
                if any(_ == 'debian/' for _ in Arcadia.list(path, as_list=True)):
                    return path
                path = Arcadia.parent_dir(path)

        raise ValueError("Can't find debian path in changed files")

    def _get_build_options(self):
        build_options = {
            'intellect': 1,
            'repositories': ','.join("yabs-{}/{}".format(u, DEFAULT_DISTRIBUTION) for u in SUPPORTED_UBUNTU_VERSIONS),
            'testing-repositories': ','.join("qabs-bamboo-{}/{}".format(u, DEFAULT_TESTING_DISTRIBUTION) for u in SUPPORTED_UBUNTU_VERSIONS),
        }

        try:
            dot_build = Arcadia.cat(self.ctx.get('ArcadiaPath') + '/.build')
            re_parser = re.compile(r"^([\w\.\-]+)\s*=\s*(.*)")
            for line in dot_build.split("\n"):
                matched = re_parser.match(line.strip())
                if matched:
                    k, v = matched.groups()
                    build_options[k.lower()] = v
        except errors.SandboxException:
            logging.info("No .build conf found")

        logging.info("Use build options:\n{}".format(build_options))

        return build_options

    def _build(self, src_dir_root, src_dir, build_options):
        version = self.ctx.get('PackageVersion')
        intellect = build_options['intellect']
        if self.ctx.get('Repositories'):
            repositories = self.ctx.get('Repositories')
        elif self.ctx.get('TestingBuild'):
            repositories = build_options['testing-repositories']
        else:
            repositories = build_options['repositories']

        revision = Arcadia.info(src_dir)['entry_revision']
        self.set_info('Build package from revision: ' + revision)

        if not version:
            version = revision
            if self.ctx.get('ArcadiaPatch'):
                version += '.' + str(self.id)
            self.ctx['PackageVersion'] = version

        if os.path.isfile(src_dir + "/debian/changelog_make"):
            logging.info("run changelog_make")
            run_process("./debian/changelog_make", work_dir=src_dir, log_prefix='changelog_make')
            intellect = 0

        for repo_distribution in repositories.split(','):
            logging.info("processing %s" % (repo_distribution))
            repo, distribution = repo_distribution.split('/')
            ubuntu_version = repo.split('-')[-1]
            if ubuntu_version not in SUPPORTED_UBUNTU_VERSIONS:
                ubuntu_version = DEFAULT_UBUNTU_VERSION

            distribution = distribution or 'unstable'
            ubuntu_version = ubuntu_version or 'precise'

            # update debian
            if os.path.isfile(src_dir + "/debian/control." + ubuntu_version):
                copy(src_dir + "/debian/control", self.abs_path() + "/control.orig")
                copy(src_dir + "/debian/control." + ubuntu_version, src_dir + "/debian/control")

            with open(src_dir + "/debian/control", 'r') as f:
                control_text = f.read()
                source_package = re.search('^Source:\s*(.*)\n', control_text).group(1)
                binary_packages = re.findall('^Package:[\t ]*(.+)$', control_text, re.MULTILINE)

            if intellect:
                copy(src_dir + "/debian/changelog", self.abs_path() + "/changelog.orig")
                with open(src_dir + "/debian/changelog", 'w') as f:
                    f.write(
                        "%s (%s) %s; urgency=low\n\n  ** package was built by sandbox task %s (YabsDebuilderDev) from %s@%s\n\n -- YABS dev. team <yabs-dev@yandex-team.ru>  %s\n" % (
                            source_package,
                            version,
                            distribution,
                            self.id,
                            self.ctx.get('ArcadiaPath'),
                            revision,
                            os.popen('date -R').read()
                        )
                    )
            else:
                with open(src_dir + "/debian/changelog", 'r') as f:
                    version, distribution = re.search('^[\.\w_-]+ \((.+?)\) (\w+)', f.read()).groups()
                self.ctx['PackageVersion'] = version

            if self.ctx.get('is_patch_check'):
                # update description for precommit builds
                self.descr = "YABS_DEBUILDER {}={} from rev.{}".format(source_package, version, revision)

            # make cmds
            build_cmd = " ".join([
                'cd ' + src_dir + ' && ',
                'pdebuild',
                '--use-pdebuild-internal',
                '' if self.ctx.get('DryRun') else '--auto-debsign --debsign-k yabs',
                '--buildresult ' + self.abs_path(),
                '--',
                '--basetgz ' + self._get_basetgz(ubuntu_version),
                '--hookdir ' + self.yabs_debuilder + '/pbuilder-hooks',
                ('--inputfile ' + self.yabs_debuilder + '/source-lists/yabs-' + ubuntu_version + '.testing.sources.list') if self.ctx.get('UseTestingSources') else '',
                '--bindmounts /run/shm --bindmounts /place/sandbox-data/build_cache --bindmounts ' + src_dir_root,
                '--buildplace ' + self.abs_path() + ' --aptcache ' + self.abs_path(),
            ])

            if self.ctx.get('DryRun'):
                process_info = run_process(build_cmd, shell=True, check=False, log_prefix='pdebuild_' + ubuntu_version)
                if process_info.returncode > 0:
                    stdout = open(process_info.stdout_path, 'r').read()
                    if 'Some index files failed to download' in stdout:
                        raise errors.TemporaryError('Temporary problems with repos')
                    else:
                        raise SandboxSubprocessError(
                            message='Build package failed',
                            stdout_path=process_info.stdout_path_filename,
                            logs_resource=getattr(getattr(channel.task, '_log_resource', None), 'id', None),
                        )
                else:
                    for binary_package in binary_packages:
                        self.set_info('Package ' + binary_package + '=' + version + ' successfully build for ' + ubuntu_version)
                    self.set_info('Skip upload because of DryRun flag')
                    continue

            version_wo_epoch = re.search('^\d+:(.*)', version)
            version_wo_epoch = version_wo_epoch.group(1) if version_wo_epoch else version
            upload_cmd = "dupload --force --to {repo} {abs_path}/{source_package}_{version_wo_epoch}_a*.changes".format(
                repo=repo,
                abs_path=self.abs_path(),
                source_package=source_package,
                version_wo_epoch=version_wo_epoch
            )

            if not self.ctx.get('CheckAlreadyUploaded') or not self._check_uploaded(binary_packages, version, repo):
                # actualy build & upload
                with GpgKey(self, "ADVQUALITY", "yabs-commiter-gpg-private", "yabs-commiter-gpg-public"):
                    process_info = run_process(build_cmd, shell=True, check=False, log_prefix='pdebuild_' + ubuntu_version)
                    if process_info.returncode > 0:
                        stdout = open(process_info.stdout_path, 'r').read()
                        if 'Some index files failed to download' in stdout:
                            raise errors.TemporaryError('Temporary problems with repos')
                        elif 'ResourceInfoError: HTTP Error 502: Bad Gateway' in stdout:
                            raise errors.TemporaryError('Temporary problems with fetch_from_sandbox')
                        else:
                            raise SandboxSubprocessError(
                                message='Build package failed',
                                stdout_path=process_info.stdout_path_filename,
                                logs_resource=getattr(getattr(channel.task, '_log_resource', None), 'id', None),
                            )
                with DebRelease(DUPLOAD_CONF):
                    with Key(self, "ADVQUALITY", "yabs-commiter-ssh"):
                        run_process(upload_cmd, shell=True, log_prefix='dupload_' + ubuntu_version)

                start_time = time.time()
                packages_were_found = False
                while not packages_were_found and time.time() < start_time + WAIT_REPO_TIMEOUT:
                    time.sleep(10)
                    logging.debug("Checking packages in repo ...")
                    packages_were_found = self._check_uploaded(binary_packages, version, repo)

                if not packages_were_found:
                    raise errors.TaskError("Packages are not found in {} after {} secs from dupload".format(repo, WAIT_REPO_TIMEOUT))

                for binary_package in binary_packages:
                    self.set_info('Package ' + binary_package + '=' + version + ' successfully uploaded to ' + repo + '/' + distribution)
            else:
                for binary_package in binary_packages:
                    self.set_info('Package ' + binary_package + '=' + version + ' is already in ' + repo)

            # restore environment
            if os.path.isfile(src_dir + "/debian/control." + ubuntu_version):
                copy(self.abs_path() + "/control.orig", src_dir + "/debian/control")
            if intellect:
                copy(self.abs_path() + "/changelog.orig", src_dir + "/debian/changelog")

            if self.ctx.get('RunBambooTests'):
                self._run_tests(binary_packages, version, ubuntu_version, build_options)

    def _check_uploaded(self, binary_packages, version, repo):
        repo_url = 'dist.yandex.ru'
        if repo.split('-')[0] in ['yabs', 'qabs', 'afpb']:
            repo_url = 'bsdist.yandex.net'

        repo_packages = []
        for distribution in ['stable', 'unstable', 'prestable', 'testing', 'configs']:
            for cpu_type in ['all', 'amd64']:
                url = 'http://' + repo_url + '/' + repo + '/' + distribution + '/' + cpu_type + '/Packages'
                try:
                    response = urlopen(url).read()
                    for chunk in response.split('\n\n'):
                        package_params = {}
                        for line in chunk.split('\n'):
                            splitted = re.compile(': ').split(line, 1)
                            if (len(splitted) == 2):
                                package_params[splitted[0]] = splitted[1]

                        if 'Package' in package_params and 'Version' in package_params:
                            repo_packages.append(package_params)
                except:
                    logging.debug("Can't call %s" % (url))

        for package in binary_packages:
            if not any(_['Package'] == package and _['Version'] == version for _ in repo_packages):
                return False

        return True

    def _get_basetgz(self, ubuntu_version):
        if self._is_testing_mode():
            return '/var/cache/pbuilder/' + ubuntu_version + '.tgz'
        else:
            items = rest.Client().resource.read(
                type='YABS_PBUILDER_IMAGE',
                state='READY',
                attrs={'release': ubuntu_version},
                limit=1
            )["items"]
            logging.debug("found: {}".format(items))

            resource_id = items[0]['id']
            return self.sync_resource(resource_id)

    def _run_tests(self, binary_packages, version, ubuntu_version, build_options):
        skip_packages = {}
        if 'parasprite-skip-packages' in build_options:
            for skip_package in build_options['parasprite-skip-packages'].split(','):
                skip_packages[skip_package] = 1

        for package in binary_packages:
            if package in skip_packages or re.search('(-ft|-pt)$', package):
                continue

            params = {
                'package': package,
                'version': version,
                'source': 'sandbox',
                'mails': self.ctx.get('precommit_check_author', self.author),
                'ubuntu_versions': ubuntu_version,
            }

            if self.ctx.get('TestIssue'):
                params.update(tickets=self.ctx.get('TestIssue'))

            build_links = bamboo_build_notify(params, build_options)
            for build_link in build_links:
                self.set_info(build_link, False)

            if ('test-rdepends-' + package) in build_options:
                for rdepend_package in build_options['test-rdepends-' + package].split(','):
                    params['package'] = rdepend_package
                    params['version'] = 'stable'
                    params['also_install'] = package + '=' + version

                    build_links = bamboo_build_notify(params, build_options)
                    for build_link in build_links:
                        self.set_info(build_link, False)

    def _is_testing_mode(self):
        return os.path.isfile('/etc/testing_sandbox')


__Task__ = YabsDebuilderDev
