import logging
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sdk2 import yav
import sandbox.sandboxsdk.parameters as sbparam
import socket
import time

SECRET_ID = "sec-01dc7bcaz5cfmtbbft03p4107b"
KEY_ID = "yt_token"
TABLE_NAME = "//home/yabs/stat/ProcessLogUpdateTime"
YT_APP_CONTROL_PATH = "//home/yabs/stat/AppControl"


# TODO: copied from supervisor
class AppControl(object):
    def __init__(self, ytc):
        self._app_control = {}
        if not ytc.exists(YT_APP_CONTROL_PATH):
            pass
        else:
            for k, v in ytc.get(YT_APP_CONTROL_PATH).items():
                v = v.lower()
                if v in ('enabled', 'paused', 'disabled'):
                    self._app_control[k] = v
                else:
                    logging.warn("Wrong app control value '{}' for '{}'".format(v, k))
                    # Treat wrong value as 'paused' as the most safe and recoverable state
                    self._app_control[k] = 'paused'

    def is_collector_can_run(self, collector_name):
        return self._app_control.get(collector_name, 'enabled') == 'enabled'

    def is_collector_enabled(self, collector_name):
        return self._app_control.get(collector_name, 'enabled') != 'disabled'


class YtClusters(sbparam.SandboxStringParameter):
    name = "yt-clusters"
    description = 'yt-clusters'
    default_value = 'freud, zeno, seneca-sas, seneca-man, seneca-vla, hahn, arnold'


class YabsDelaySbytStatGraphiteSender(SandboxTask):
    type = 'YABS_DELAY_SBYT_STAT'
    environment = (
        environments.PipEnvironment("yandex-yt", use_wheel=True),
    )

    input_parameters = [
        YtClusters
    ]

    cores = 1

    GRAPHITE_HOSTS = [
        "mega-graphite-man.search.yandex.net:2024",
        "mega-graphite-sas.search.yandex.net:2024"
    ]

    def send_points_to_graphite(self, str_for_graphite):
        for server in self.GRAPHITE_HOSTS:
            (host, port) = server.split(":")
            for i in xrange(0, 5):
                try:
                    sock = socket.create_connection((host, port), 1)  # 1 sec timeout
                    sock.sendall(str_for_graphite)
                    sock.close()
                    break
                except:
                    logging.info("Retry {}".format(i))
            else:
                logging.info("Connect to {} failed".format(host))

    def send_data_from_one_cluster(self, token, yt_cluster):
        import yt.wrapper as yt
        try:
            logging.info("connect to " + str(yt_cluster))
            cfg = {
                "tabular_data_format": yt.JsonFormat(control_attributes_mode="row_fields"),
                "detached": False,
                "token": token,
                "proxy": {"url": yt_cluster}
            }
            ytc = yt.YtClient(config=cfg)

            logging.info("Select data from {:s}".format(yt_cluster))
            rows = ytc.select_rows("""
                    CollectorName,
                    LogType,
                    UpdateTime
                from [{:s}]
            """.format(TABLE_NAME))
        except Exception as e:
            logging.warn("Can't connect or select to {:s}: ".format(yt_cluster) + str(e))
            return

        logging.info("Generate rows for graphite")
        cur_time = int(time.time())
        rows_for_graphite = ""
        app_control = AppControl(ytc)
        for row in rows:
            if not app_control.is_collector_enabled(row["CollectorName"]):
                continue
            rows_for_graphite += " ".join([
                ".".join(["one_min", yt_cluster, row["CollectorName"], row["LogType"], "delay"]),
                str(max(cur_time - row["UpdateTime"], 0)),  # its impossible situation, when cur_time - row["UpdateTime"] < 0, but how know...
                str(cur_time)
            ]) + "\n"
        logging.info(rows_for_graphite)
        self.send_points_to_graphite(rows_for_graphite)

    def on_execute(self):
        yt_clusters = self.ctx[YtClusters.name]
        yt_clusters = yt_clusters.replace(" ", "")  # delete space
        logging.info("All Yt Clusters: " + yt_clusters)
        yt_clusters = yt_clusters.split(",")  # split by comma

        token = yav.Secret(SECRET_ID).data()[KEY_ID]
        for yt_cluster in yt_clusters:
            self.send_data_from_one_cluster(token, yt_cluster)


__Task__ = YabsDelaySbytStatGraphiteSender
