# -*- coding: utf-8 -*-
import json
import logging
from sandbox import sdk2
from sandbox.common.types.task import ReleaseStatus
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.resource as ctr


RESOURCE_PATH = 'duty.json'


class YabsDutyScheduleGeneratorBinary(sdk2.Resource):
    """ Generilka dejurstv binary"""
    releasable = True
    any_arch = False
    auto_backup = True
    executable = True
    releasers = ['danilgrig', 'bahbka', 'tudalova', 'fortuna', 'abordonos', 'astyts', 'violettfaid', 'void2', 'torubarov', 'kacher', 'coolmanool', 'robot-yabs-infra']


class YabsDutySchedule(sdk2.Resource):
    """ Timetable of duty"""
    auto_backup = True
    week = sdk2.parameters.String("week")


class YabsDutyScheduleGenerator(sdk2.Task):

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        commit_to_table = sdk2.parameters.Bool("Commit duty to table")
        weeks_count = sdk2.parameters.Integer("Weeks to generate", default=1)
        vault_token_path = sdk2.parameters.String("Vault key token name")
        generator_binary = sdk2.parameters.Resource(
            'Generilka binary. Stable resource by default',
            resource_type=YabsDutyScheduleGeneratorBinary,
            required=False,
        )
        create_resources_with_duty = sdk2.parameters.Bool("Create resources with duty for release machine")

    def create_resources_with_duty(self):
        with open(RESOURCE_PATH) as f:
            duty_data = json.load(f)
        for num, week in enumerate(duty_data):
            week_data_path = "{}.json".format(num)
            with open(week_data_path, 'w') as week_data_file:
                week['Week'] = week['Week'][0]
                json.dump(week, week_data_file, indent=4)
                YabsDutySchedule(self, 'Timetable of duty', week_data_path, ttl=42, week=week['Week'])

    def get_executable(self):
        if self.Parameters.generator_binary:
            res = self.Parameters.generator_binary
        else:
            attrs = {
                'released': ReleaseStatus.STABLE,
            }

            res = YabsDutyScheduleGeneratorBinary.find(state=ctr.State.READY, attrs=attrs).order(-sdk2.Resource.id).first()
        res_data = sdk2.ResourceData(res)
        return str(res_data.path)

    def on_execute(self):
        token = sdk2.Vault.data("YABS-YTSTAT", self.Parameters.vault_token_path)
        command = [
            self.get_executable(),
            '--token',
            token,
            '--weeks',
            str(self.Parameters.weeks_count),
        ]
        if self.Parameters.commit_to_table:
            command.append('-P')
            command.append('1')

        if self.Parameters.create_resources_with_duty:
            command.append('-s')
            command.append(RESOURCE_PATH)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("YabsDutyScheduleGenerator")) as pl:
            sp.check_call(command, stdout=pl.stdout, stderr=sp.STDOUT)

        if self.Parameters.create_resources_with_duty:
            self.create_resources_with_duty()
