# -*- coding: utf-8 -*-

import logging

from sandbox.common.types.client import Tag

from sandbox.sdk2 import yav
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common import utils as sb_utils
from sandbox.projects.common.yabs.juggler import (
    ServiceStatus, Juggler,
    STATUS_OK, STATUS_WARN, STATUS_CRIT,
)
from sandbox.projects.common.yabs.lbyt_reader.job_yt_client import (
    MANAGER_STATE, READER_STATE, NAMESPACE as LBYT_NAMESPACE
)
from sandbox.projects.common.yabs.lbyt_reader.config import MANAGER_JOBS, YAV_SECRET_ID

JOBS_SERVICE_NAME = 'lbyt_jobs'


class YtPathPrefix(SandboxStringParameter):
    name = 'yt_path_prefix'
    description = 'Common prefix for all Yt paths'
    default_value = '//home/yabs/lbyt-reader-testing'


class YabsLbYtJobsMonitoring(SandboxTask):
    type = 'YABS_LBYT_JOBS_MONITORING'

    input_parameters = (
        YtPathPrefix,
    )
    environment = (
        environments.PipEnvironment('requests'),
        environments.PipEnvironment('yandex-yt', '0.8.38a1', use_wheel=True),
    )
    client_tags = Tag.LINUX_PRECISE
    cores = 1

    @staticmethod
    def make_status_from_tuples(host, service, status_tuples):
        def status_tuples_order(status_tuple):
            return [STATUS_CRIT, STATUS_WARN, STATUS_OK].index(status_tuple[0])

        if not status_tuples:
            return ServiceStatus(host, service, STATUS_OK)
        status_tuples.sort(key=status_tuples_order)
        description = '; '.join(status_tuple[1] for status_tuple in status_tuples)
        return ServiceStatus(host, service, status_tuples[0][0], description)

    def on_execute(self):
        statuses = []

        yt_clusters = self.load_jobs()
        logging.info('Found jobs on yt clusters %s', ', '.join(sorted(yt_clusters)))

        for yt_cluster, jobs in yt_clusters.iteritems():
            statuses.append(self.get_yt_cluster_status(yt_cluster, jobs))

        logging.debug('Sending to juggler:\n\t%s', '\n\t'.join(map(str, statuses)))
        Juggler().send(statuses)

    def load_jobs(self):
        yt_clusters = {}
        for job, config in MANAGER_JOBS.iteritems():
            for yt_cluster in config['yt']:
                if yt_cluster not in yt_clusters:
                    yt_clusters[yt_cluster] = set()
                yt_clusters[yt_cluster].add(job)
        return yt_clusters

    def get_yt_cluster_status(self, yt_cluster, jobs):
        cluster_statuses = []

        jobs_info = self.get_jobs_info(yt_cluster)
        if jobs_info is None:
            cluster_statuses.append((STATUS_WARN, 'Failed to get jobs info'))
            jobs_info = {}

        for job, state in jobs_info.iteritems():
            if job not in jobs:
                logging.info('Job "%s" on %s was not found in config', job, yt_cluster)
                cluster_statuses.append(
                    (STATUS_WARN, 'Job "{}" was not found in config'.format(job))
                )
                continue
            if state.get(MANAGER_STATE) != state.get(READER_STATE):
                desc = 'Different states of manager and reader for job "{}"'.format(job)
                logging.info('%s', '{} on {}'.format(desc, yt_cluster))
                cluster_statuses.append((STATUS_WARN, desc))

        cluster_host = 'sbyt-{}.yt.yandex.net'.format(yt_cluster)
        return self.make_status_from_tuples(cluster_host, JOBS_SERVICE_NAME, cluster_statuses)

    def get_jobs_info(self, yt_cluster):
        from yt.wrapper.yson import YsonEntity
        target_path = '/'.join([
            sb_utils.get_or_default(self.ctx, YtPathPrefix).strip(),
            LBYT_NAMESPACE,
            'control',
        ])
        result = None
        try:
            yt = self.get_yt_connection(yt_cluster)
            if yt.exists(target_path):
                result = yt.get(target_path)
                for element in result:
                    if isinstance(result[element], YsonEntity):
                        result[element] = yt.get('{}/{}'.format(target_path, element))
        except Exception as err:
            logging.warn(
                'Failed to get jobs info from %s on yt cluster %s: %s',
                target_path, yt_cluster, err,
            )
        return result

    def get_yt_connection(self, cluster):
        from yt.wrapper import YtClient
        token = yav.Secret(YAV_SECRET_ID).data()["yt"]
        return YtClient(proxy=cluster, token=token, config={'api_version': 'v3'})


__Task__ = YabsLbYtJobsMonitoring
