# -*- coding: utf-8 -*-
import logging
import smtplib
import time
import datetime
from sandbox import common
import sandbox.sandboxsdk.parameters as sbparam
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.task import SandboxTask
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from collections import defaultdict

VAULT_USER = "ADVQUALITY"
VAULT_TOKEN = "yt-token-report-for-ssp"
SECONDS_IN_ONE_DAY = 60 * 60 * 24
FIELDS = [
    "Date",
    "Hits",
    "Bids",
    "SSPWins",
    "ImpressionPrice",
    "PartnerPrice"
]
SUFFIX = {
    "Date": "",
    "Hits": "",
    "Bids": "",
    "SSPWins": "",
    "ImpressionPrice": "&#36",
    "PartnerPrice": "&#36"
}
CNT_FIELDS = len(FIELDS) - 1


class ReportForSspConnectError(Exception):
    pass


class SendException(Exception):
    pass


class YtClusters(sbparam.SandboxStringParameter):
    name = "yt-clusters"
    description = 'yt-clusters'
    default_value = 'seneca-fin, seneca-sas'


class EmailRecipient(sbparam.SandboxStringParameter):
    name = "Email Recipient"
    description = 'Email Recipient'
    default_value = 'skrrydg, ssp-int'


class YabsReportForSsp(SandboxTask):
    type = 'YABS_REPORT_FOR_SSP'
    environment = (
        environments.PipEnvironment("yandex-yt", use_wheel=True),
    )

    input_parameters = [
        YtClusters,
        EmailRecipient
    ]

    def _get_currency_id(self, ytc, currency):
        rows = ytc.select_rows(
            """
                CurrencyID
            from
                [//yabs/Dicts/Currency]
            where
                CurrencyCode = '""" + str(currency) + "'"
        )

        currency_id = None
        for row in rows:
            currency_id = row["CurrencyID"]

        return currency_id

    def _get_rate(self, ytc, currency, current_time):
        currency_id = self._get_currency_id(ytc, currency)
        rows = ytc.select_rows("""
                Rate
            from
                [//yabs/Dicts/CurrencyRates]
            where
                    CurrencyID = """ + str(currency_id) + """
                and
                    EventDate <= """ + str(current_time) + """
            order by
                    -EventDate
            limit 1
            """
        )

        rate = None
        for row in rows:
            rate = float(row["Rate"]) / 1000000.0

        logging.info(str(rate))
        return rate

    def _get_stats(self, ytc, start_time, end_time):
        rate_usd = self._get_rate(ytc, "USD", end_time)

        rows = ytc.select_rows("""
                SSPID,
                sum(RtbAllHits) as Hits,
                sum(WinHits) as Bids,
                sum(SSPWins) as SSPWins,
                int64(double(sum(SSPPrice)) / 1000.0 / 1000.0 / """ + str(rate_usd) + """) as ImpressionPrice,
                int64(double(sum(PartnerPrice)) / 1000.0 / 1000.0 / """ + str(rate_usd) + """) as PartnerPrice
            from
                [//yabs/SSPCraterStat]
            where
                    UpdateTime < """ + str(end_time) + """
                and
                    UpdateTime >= """ + str(start_time) + """
            group by
                SSPID
        """)

        array_rows = []
        for row in rows:
            row["Date"] = datetime.datetime.fromtimestamp(int(start_time)).strftime('%Y-%m-%d')
            array_rows.append(row)
        return array_rows

    def _is_time_field(self, string):
        return (string.find("-") != -1)

    def _add_separators(self, string, separator):
        if (self._is_time_field(string)):
            return string

        sign = ""
        if (string[0] == "+" or string[0] == '-'):
            sign = string[0]
            string = string[1:]

        pos_r = len(string)
        pos_l = max(len(string) - 3, 0)
        string_with_separator = ""
        while (pos_l > 0):
            string_with_separator = separator + string[pos_l:pos_r] + string_with_separator
            pos_l = max(pos_l - 3, 0)
            pos_r = max(pos_r - 3, 0)

        string_with_separator = string[pos_l:pos_r] + string_with_separator
        return sign + string_with_separator

    def _get_distinct_ssp(self, ssp_stat):
        distinct_ssp = []
        for row in ssp_stat[-1]:
            distinct_ssp.append(row["SSPID"])
        return distinct_ssp

    def _get_tables_for_ssp(self, ssp_stat, ssp):
        result_string = ""
        result_sum_string = ""

        sum_stats = defaultdict(int)
        for day_number, rows_per_day in enumerate(ssp_stat):
            find = 0
            for row in rows_per_day:
                if (row["SSPID"] == ssp):
                    cur_result_string = ""
                    for field in FIELDS:
                        if (row[field] is None):
                            cur_result_string += "<td>" + self._add_separators(str(0), ".") + str(SUFFIX[field]) + "</td>"
                        else:
                            cur_result_string += "<td>" + self._add_separators(str(row[field]), ".") + str(SUFFIX[field]) + "</td>"
                    for field in FIELDS[1:]:
                        if not (row[field] is None):
                            sum_stats[field] += int(row[field])
                    result_string += "<tr>" + str(cur_result_string) + "</tr>"
                    find = 1
                    break
            if (not find):
                cur_date = datetime.datetime.now().replace(day=len(ssp_stat) - day_number).strftime('%Y-%m-%d')
                result_string += "<tr> <td>" + str(cur_date) + "</td>" + "<td> 0 </td>" * CNT_FIELDS + "</tr>"

        sum_stats["Date"] = (datetime.datetime.now() - datetime.timedelta(days=1)).strftime('%Y-%m')
        for field in FIELDS:
            result_sum_string += "<td>" + self._add_separators(str(sum_stats[field]), ".") + str(SUFFIX[field]) + "</td>"
        result_sum_string = "<tr>" + result_sum_string + "</tr>"

        return result_string, result_sum_string

    def _get_title_for_ssp(self, ytc, ssp_id):
        rows = ytc.select_rows("""
                Title
            from
                [//yabs/Dicts/SSPInfo]
            where
                SSPID = """ + str(ssp_id) + """
        """)

        title = ""
        for row in rows:
            title = row["Title"]

        return title

    def _send_mails(self, ytc, ssp_stat, email_recipient):
        distinct_ssp = self._get_distinct_ssp(ssp_stat)
        logging.info("DIST SSP:")
        logging.info(distinct_ssp)
        for ssp in distinct_ssp:
            table_for_current_ssp, sum_table_for_current_ssp = self._get_tables_for_ssp(ssp_stat, ssp)
            ssp_title = self._get_title_for_ssp(ytc, ssp)

            msg = MIMEMultipart('alternative')
            msg.set_charset('utf8')
            msg['From'] = 'Report for SSP <yabs-scheduler@yandex-team.ru>'
            msg['To'] = ", ".join(email_recipient)
            msg['Subject'] = str(ssp_title)

            html = """
                <html>
                <head>
                    <style>
                        table {
                            font-family: arial, sans-serif;
                            border-collapse: collapse;
                        }

                        td, th {
                            border: 1px solid #dddddd;
                            text-align: left;
                            padding: 8px;
                        }

                        tr:nth-child(even) {
                            background-color: #dddddd;
                        }
                    </style>
                </head>
                <body>
                    <table>
                        <tr>""" + " ".join(map(lambda x: "<th>" + str(x) + "</th>", FIELDS)) + """
                        </tr>""" + str(sum_table_for_current_ssp) + """
                    </table>
                    <br>
                    <table>
                        <tr>""" + " ".join(map(lambda x: "<th>" + str(x) + "</th>", FIELDS)) + """
                         </tr>""" + str(table_for_current_ssp) + """
                    </table>
                </body>
                </html>
            """
            logging.info(html)
            html_text = MIMEText(html, 'html', 'utf8')
            msg.attach(html_text)

            try:
                srv = smtplib.SMTP('yabacks.yandex.ru', port=25)
                srv.sendmail('Report for SSP <yabs-scheduler@yandex-team.ru>', email_recipient, msg.as_string())
                srv.quit()
            except smtplib.SMTPException:
                raise SendException('No e-mails was sent. Internal SMTP exception occured')

    def _yt_connect(self):
        import yt.wrapper as yt

        yt_clusters = self.ctx[YtClusters.name]
        logging.info("All Yt Clusters: " + yt_clusters)
        yt_clusters = yt_clusters.replace(" ", "")  # delete space
        yt_clusters = yt_clusters.split(",")  # split by comma

        logging.info("Connect to Yt")

        ytc = None
        successful_connection = False
        for yt_cluster in yt_clusters:
            if successful_connection:
                break

            try:
                logging.info("Try connect to " + str(yt_cluster))
                cfg = {
                    "tabular_data_format": yt.JsonFormat(control_attributes_mode="row_fields"),
                    "detached": False,
                    "token": self.get_vault_data(VAULT_USER, VAULT_TOKEN),
                    "proxy": {"url": yt_cluster},
                }
                ytc = yt.YtClient(config=cfg)
                logging.info("connect to " + str(yt_cluster))
                successful_connection = True
            except:
                pass

        if not successful_connection:
            raise ReportForSspConnectError("Can't connect to any cluster")

        return ytc

    def on_execute(self):
        # extract email recipient
        email_recipient = self.ctx[EmailRecipient.name]
        email_recipient = email_recipient.replace(" ", "")  # delete space
        email_recipient = email_recipient.split(",")  # split by comma

        for i, cur_recipient in enumerate(email_recipient):
            if (cur_recipient.find('@') == -1):
                email_recipient[i] = cur_recipient + "@yandex-team.ru"

        logging.info("Email Recipients: " + ", ".join(email_recipient))

        yesterday_date = datetime.datetime.now() - datetime.timedelta(days=1)
        yesterday_date = yesterday_date.replace(hour=0, minute=0, second=0, microsecond=0)
        day_number = yesterday_date.day

        try:
            sandbox = common.rest.Client()
            prev_task_id = sandbox.task.read(scheduler=self.scheduler, order="-id", limit=2)["items"][1]["id"]
            last_day_number = sandbox.task[prev_task_id].context.read()["day_number"]
            is_success = sandbox.task[prev_task_id].context.read()["last_success"]
            if (day_number == last_day_number and is_success):
                logging.info("We already send report today")
                return
            else:
                logging.info("We don't send report today yet")
        except:
            logging.info("It's first sheduller's task")
            logging.info("We don't send report today yet")

        self.ctx["last_success"] = 0
        self.ctx["day_number"] = day_number

        logging.info("Connect to Yt")
        ytc = self._yt_connect()

        ssp_stat = []
        logging.info("Yesterday - " + str(day_number) + " day in month")
        start_time = int(time.mktime(yesterday_date.replace(day=day_number).timetuple()))
        end_time = start_time + SECONDS_IN_ONE_DAY

        for i in range(0, day_number):
            logging.info("start_time: " + str(start_time) + "\t" + "end_time: " + str(end_time))
            ssp_stat.append(self._get_stats(ytc, start_time, end_time))
            start_time -= SECONDS_IN_ONE_DAY
            end_time -= SECONDS_IN_ONE_DAY

        self._send_mails(ytc, ssp_stat, email_recipient)
        self.ctx["last_success"] = 1


__Task__ = YabsReportForSsp
