# -*- coding: utf-8 -*-
import datetime
import json
import logging

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.common.utils import get_or_default
from sandbox.projects.common.yabs.server.db import yt_bases, utils as dbutils
from sandbox.projects.yabs.qa.resource_types import YABS_CS_INPUT_SPEC
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolGroupParameter
from sandbox.sandboxsdk.task import SandboxTask


class Options(SandboxBoolGroupParameter):
    name = 'options'
    description = 'Options:'
    SWITCH_TESTENV = 'switch_testenv_bases'
    choices = [
        ('Switch testenv bases to new resource', SWITCH_TESTENV),
    ]
    default_value = ''


class YtProxyOption(SandboxStringParameter):
    name = "yt_proxy"
    description = name
    default_value = "hahn"


class ArchivesRoot(SandboxStringParameter):
    name = "archives_root"
    description = 'Root node for import archives'
    required = True
    default_value = '//home/yabs-cs-sandbox/input-archive'


class TestenvSwitchTriggerValue(sdk2.parameters.String):
    name = 'testenv_switch_trigger_value'
    description = 'Use following string as testenv switch trigger value (backup date will be used if empty)'
    default_value = ''


class YabsServerArchiveCSInput(SandboxTask):
    type = 'YABS_SERVER_ARCHIVE_CS_INPUT'
    description = 'Build cs_import from head (if needed), run ./cs_import --print-info and archive input tables'

    input_parameters = [YtProxyOption, Options, ArchivesRoot, TestenvSwitchTriggerValue]

    execution_space = 4 * 1024

    environment = (
        PipEnvironment('yandex-yt', use_wheel=True),
    )

    client_tags = Tag.LINUX_PRECISE
    cores = 1

    def has_option(self, opt):
        return opt in self.ctx.get(Options.name, '').split()

    def on_execute(self):
        from yt.wrapper import YtClient, ypath_join

        token = dbutils.get_yabscs_yt_token(self)
        proxy = self.ctx.get(YtProxyOption.name, 'hahn')
        logging.info("YT proxy: %s", proxy)

        client = YtClient(proxy=proxy, token=token)

        archive_root = ypath_join(get_or_default(self.ctx, ArchivesRoot), str(self.id))
        client.create("map_node", archive_root, ignore_existing=True, recursive=True)

        yt_bases.renew_expiration_time(client, archive_root, yt_bases.DEFAULT_CS_INPUT_ARCHIVE_TTL)

        spec = {
            yt_bases.AUXILLARY_DATA_IN_SPEC_KEY: {
                yt_bases.ARCHIVE_ROOT_KEY: archive_root,
                yt_bases.ARCHIVE_ROOT_TTL_KEY: yt_bases.DEFAULT_CS_INPUT_ARCHIVE_TTL,
            }
        }
        self._create_spec_resource(spec, proxy)

    def _create_spec_resource(self, archive_spec, proxy):
        archive_spec['proxy'] = proxy

        spec_path = 'input_spec.json'
        with open(spec_path, 'w') as out:
            json.dump(archive_spec, out, indent=4)

        attrs = dict(yt_proxy=proxy)
        if self.has_option(Options.SWITCH_TESTENV):
            testenv_switch_trigger_value = get_or_default(self.ctx, TestenvSwitchTriggerValue) or datetime.date.today().strftime('%Y%m%d')
            attrs['testenv_switch_trigger'] = testenv_switch_trigger_value

        self.create_resource(
            description="yabs-cs cs_import input spec for Testenv bases",
            resource_path=spec_path,
            resource_type=YABS_CS_INPUT_SPEC,
            attributes=attrs,
            arch='any'
        )


__Task__ = YabsServerArchiveCSInput
