import os
import logging
import datetime
import subprocess
import json

from collections import defaultdict
from concurrent import futures

from sandbox.common.errors import TemporaryError

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter, ResourceSelector, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process


from sandbox.projects.common.yabs.server.db.task.mysql import install_yabs_mysql, use_myisampack
from sandbox.projects.common.yabs.task import TaskWrapper  # XXX: ditch this!

from sandbox.projects.common.yabs.server.parameters import BackupDate

from sandbox.projects.yabs.qa.resource_types import (
    YABS_MYSQL_ARCHIVE_CONTENTS,
    YABS_MYSQL_RESTORE_DESCRIPTION,
    MYSQL_TABLE_ARCHIVE,
)

FIX_DEBUG_COOKIE = ('yabs', 'yabsdb', 'RecordSign', "INSERT IGNORE INTO yabsdb.RecordSign(Sequence, SignTime, Sign) VALUES(151917, '2017-02-17 23:00:00', 99007637);")


class TableDescrResource(ResourceSelector):
    resource_type = YABS_MYSQL_RESTORE_DESCRIPTION
    name = 'table_descr_resource'
    description = 'Resource that describes what to restore.'


class TbListArcPath(SandboxArcadiaUrlParameter):
    name = 'tb_list_arc_path'
    default_value = ''
    description = 'Path in arcadia to table list (in JSON format). Example: arcadia:/arc/trunk/arcadia/yabs/utils/experiment-switcher/used_tables.json'
    required = False


class SourceHost(SandboxStringParameter):
    name = 'source_host'
    default_value = 'bsbackup1-02.yabs.yandex.ru'
    description = 'MySQL host'
    required = True


class YabsServerB2BPrepareSQL(TaskWrapper):
    execution_space = 1024 * 1024

    privileged = True

    type = 'YABS_SERVER_B2B_PREPARE_SQL'

    environment = [
        environments.PipEnvironment('MySQL-python', '1.2.5', use_wheel=True),
    ]

    input_parameters = (BackupDate, TableDescrResource, TbListArcPath, SourceHost)

    restore_descr_key = TableDescrResource.name
    output_resource_type = MYSQL_TABLE_ARCHIVE

    def on_execute(self):
        backup_date = self.ctx.get(BackupDate.name)
        if not backup_date.startswith('20'):
            backup_date = (datetime.datetime.now() - datetime.timedelta(hours=3)).strftime("%Y%m%d")

        instance, backup, tables_by_database = self._get_restore_descr(backup_date)
        if backup is None:
            self.set_info("Nothing to do")
            return

        install_yabs_mysql(dbrestore=True, mysql_instances={'yabs'})

        for database, tables in tables_by_database.iteritems():
            cmdline = [
                '/usr/bin/yabs-dbrestore',
                '--lock-timeout', '3600',
                '--database', database,
                '-t', ','.join(tables),
                '--backup', backup,
                '--date', backup_date,
            ]
            if backup != 'bsdb':
                cmdline += ['--source-hosts', self.ctx[SourceHost.name]]
            logging.info('Running %s', ' '.join(cmdline))
            try:
                run_process(cmdline, log_prefix="yabs-dbrestore-{}".format(database))
            except Exception as err:
                raise TemporaryError(str(err))

        _patch_tables(backup, tables_by_database)

        subprocess.check_call(['chmod -R a+rw /var/lib/mysql.*/'], shell=True)  # Do we really need this??

        archive_contents = self._archive_tables(instance, tables_by_database, backup_date)
        ac_res_path = 'archive_contents.json'
        with open(ac_res_path, 'w') as ac_file:
            json.dump(archive_contents, ac_file)

        self.create_resource(
            description="Contents fragment for {}".format(backup_date),
            resource_path=ac_res_path,
            resource_type=YABS_MYSQL_ARCHIVE_CONTENTS,
            attributes={'date': backup_date},
            arch='any'
        )

    def _get_restore_descr(self, backup_date):
        tables_by_database = defaultdict(set)
        backups = set()
        instances = set()

        # First from resource
        descr_res_id = self.ctx.get(TableDescrResource.name)
        if descr_res_id:
            descr_path = self.sync_resource(descr_res_id)
            td_data = json.load(open(descr_path))
            for key, source in td_data.iteritems():
                if source.startswith('='):
                    continue
                inst, db, table = key.split('.')
                backups.add(source)
                instances.add(inst)
                tables_by_database[db].add(table)

        # Then from Arcadia file
        tb_list_file = self.ctx.get(TbListArcPath.name, '')
        if tb_list_file:
            Arcadia.export(self.ctx.get(TbListArcPath.name, ''), 'tb_list.json')
            tl = json.loads(open('tb_list.json').read())
            for db, tables in tl.iteritems():
                tables_by_database[db] |= set(tables)
            backups.add('bsdb')
            instances.add('yabs')

        if len(instances) > 1:
            raise RuntimeError("Cannot restore tables from multiple instances in one task")
        if len(backups) > 1:
            raise RuntimeError("Cannot restore tables from multiple backups in one task")
        if backups:
            return instances.pop(), backups.pop(), tables_by_database
        return None, None, {}

    def _archive_tables(self, instance, tables_by_database, date):

        with futures.ThreadPoolExecutor(max_workers=24) as pool:
            fts = []
            for db, tables in tables_by_database.iteritems():
                for table in tables:
                    fts.append(pool.submit(archive_table, instance, db, table))

            table_count = len(fts)

            out = dict()
            uncompressed_sizes = dict()

            for n, ft in enumerate(futures.as_completed(fts)):
                inst, db, tname, path, size = ft.result()

                logging.info("Table %s.%s.%s archived (%s of %s)", inst, db, tname, n, table_count)

                attributes = {'instance': inst, 'db': db, 'table': tname, 'date': date}
                res = self.create_resource(
                    description="{}.{}.{} table archive ({})".format(inst, db, tname, date),
                    resource_path=path,
                    resource_type=self.output_resource_type,
                    attributes=attributes,
                    arch='any'
                )
                out['.'.join((inst, db, tname))] = res.id
                uncompressed_sizes[res.id] = size

        return {'tables': out, 'sizes': uncompressed_sizes}


def _patch_tables(backup, tables_by_database):
    if backup != 'bsdb':
        return
    try:
        tables = tables_by_database['yabsdb']
    except KeyError:
        return

    parts = []
    if 'OrderInfo' in tables:
        parts += [
            "UPDATE yabsdb.OrderInfo set StopTime= '2010-10-10 23:58:31' WHERE Complete=1 and StopTime >= NOW() - INTERVAL 7 DAY;",
            "UPDATE yabsdb.OrderInfo set EditTime=NOW() where EditTime>NOW();",
        ]
    if 'Constant' in tables:
        parts.append("INSERT IGNORE INTO yabsdb.Constant (Name, Value) VALUES ('phraseprice-deleted-days', 2000);")
    if FIX_DEBUG_COOKIE[2] in tables:
        parts.append(FIX_DEBUG_COOKIE[3])
    parts.append('FLUSH TABLES;')
    query = ''.join(parts)
    run_process(['mysql', '-u', 'root', '-b', 'yabsdb', '-v', '-v', '-v', '-e', query], log_prefix='mysql_patch_tables')


def archive_table(inst, db, tname):
    try:
        return _archive_table(inst, db, tname)
    except Exception:
        logging.exception("Failed to archive %s.%s.%s", inst, db, tname)
        raise


def _archive_table(inst, db, tname):
    tar_path = '.'.join([inst, db, tname, 'tar.gz'])
    logging.info("Creating %s...", tar_path)

    TABLES_ROOT = '/var/lib/mysql.yabs/'
    subpaths = []
    paths = []
    total_nonpacked_size = 0

    for ext in ['frm', 'MYD', 'MYI']:
        subpath = '{}/{}.{}'.format(db, tname, ext)
        subpaths.append(subpath)

        path = TABLES_ROOT + subpath
        paths.append(path)
        total_nonpacked_size += os.stat(path).st_size

        if ext.upper() == 'MYI' and use_myisampack(tname):
            logging.debug("Running myisampack on %s", path)
            subprocess.check_call(['sudo', 'myisampack', path])
            logging.debug("Running myisamchk -rq on %s", path)
            subprocess.check_call(['sudo', 'myisamchk', '-rq', path])

    cmdline = ['tar', '-czf', tar_path, '-C', TABLES_ROOT] + subpaths
    logging.debug("Running %s", ' '.join(cmdline))
    subprocess.check_call(cmdline)
    for path in paths:
        os.unlink(path)

    return inst, db, tname, tar_path, total_nonpacked_size


__Task__ = YabsServerB2BPrepareSQL
