from collections import defaultdict

from sandbox.common import rest

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import ResourceSelector

from sandbox.projects.yabs.qa.resource_types import YABS_SERVER_B2B_BINARY_BASE
from sandbox.projects.yabs.bases.keys import (
    CHKDB_CTX_KEY,
    CHKDB_RESOURCE_ATTR_KEY,
    CHKDB_WITHOUT_HASH_CTX_KEY,
    CHKDB_WITHOUT_HASH_RESOURCE_ATTR_KEY,
    DB_SIZE_BY_ROLE,
    DB_SIZE_CTX_KEY,
    UNPACKED_SIZE_ATTR_KEY,
)

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace, trace_entry_point


class DBResourceList(ResourceSelector):
    """Binary base list parameter. """
    name = 'db_resource_list'
    multiple = True
    resource_type = YABS_SERVER_B2B_BINARY_BASE
    description = "Binary bases"


class DBResourceListBs(ResourceSelector):
    """Binary base list parameter for bs """
    name = 'db_resource_list_bs'
    multiple = True
    resource_type = YABS_SERVER_B2B_BINARY_BASE
    description = "Binary bases for bs"


class DBResourceListBsrank(ResourceSelector):
    """Binary base list parameter for bsrank """
    name = 'db_resource_list_bsrank'
    multiple = True
    resource_type = YABS_SERVER_B2B_BINARY_BASE
    description = "Binary bases for bsrank"


class DBResourceListYabs(ResourceSelector):
    """Binary base list parameter for yabs """
    name = 'db_resource_list_yabs'
    multiple = True
    resource_type = YABS_SERVER_B2B_BINARY_BASE
    description = "Binary bases for yabs"


FIELDS_TO_ROLE = {
    DBResourceListBs.name: 'bs',
    DBResourceListBsrank.name: 'bsrank',
    DBResourceListYabs.name: 'yabs',
}


class YabsServerDBSizeAggregate(SandboxTask):

    type = 'YABS_SERVER_DB_SIZE_AGGREGATE'
    description = 'Dumb task, just aggregates binary base sizes'

    execution_space = 16  # We work with task contexts and create one very small resource
    cores = 1

    input_parameters = [DBResourceList, DBResourceListBs, DBResourceListBsrank, DBResourceListYabs]

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        cl = rest.Client()
        db_size_by_type = dict()
        chkdb_by_type = dict()
        chkdb_without_hash_by_type = dict()
        db_size_by_role = {
            role: 0
            for role in FIELDS_TO_ROLE.values()
        }

        all_resources = set(self.ctx.get(DBResourceList.name, []) or [])
        resources_to_roles = defaultdict(set)
        for field, role in FIELDS_TO_ROLE.iteritems():
            resources = self.ctx.get(field, []) or []
            all_resources.update(resources)
            for res_id in resources:
                resources_to_roles[res_id].add(role)

        for res_id in all_resources:
            with trace('read raw_attrs', info=dict(res_id=res_id)):
                raw_attrs = cl.resource[res_id].attribute.read()
            attrs = {a['name']: a['value'] for a in raw_attrs}
            try:
                db_type = attrs.get('type') or attrs['tag']
                db_size = int(attrs[UNPACKED_SIZE_ATTR_KEY])
                db_chkdb = attrs.get(CHKDB_RESOURCE_ATTR_KEY)
                db_chkdb_without_hash = attrs.get(CHKDB_WITHOUT_HASH_RESOURCE_ATTR_KEY)
            except KeyError as err:
                raise SandboxTaskFailureError("Resource %s has no attribute %s" % (res_id, err))
            chkdb_by_type[db_type] = db_chkdb
            chkdb_without_hash_by_type[db_type] = db_chkdb_without_hash
            db_size_by_type.setdefault(db_type, []).append(db_size)  # usually we have one base for one type
            for role in resources_to_roles[res_id]:
                db_size_by_role[role] += db_size
        avg_sizes = {db_type: str(sum(db_sizes) / len(db_sizes)) for db_type, db_sizes in db_size_by_type.iteritems()}
        self.ctx[DB_SIZE_CTX_KEY] = avg_sizes
        self.ctx[DB_SIZE_BY_ROLE] = db_size_by_role
        self.ctx[CHKDB_CTX_KEY] = chkdb_by_type
        self.ctx[CHKDB_WITHOUT_HASH_CTX_KEY] = chkdb_without_hash_by_type


__Task__ = YabsServerDBSizeAggregate
