# -*- coding: utf-8 -*-
import logging
from sandbox import sdk2

from sandbox.projects.YabsServerDBSizeAggregate import DBResourceList
from sandbox.projects.yabs.bases.keys import DB_SIZE_CTX_KEY, CHKDB_CTX_KEY, CHKDB_WITHOUT_HASH_CTX_KEY
from sandbox.projects.yabs.qa.compare import compare_chkdb, compare_db_sizes
from sandbox.projects.yabs.qa.resource_types import YabsDbSmartMetaReport
from sandbox.projects.common import file_utils as fu
from sandbox.projects.yabs.qa.tasks.base_compare_task.task import create_input_parameters, BaseCompareTask
from sandbox.projects.yabs.qa.tasks.base_compare_task.parameters import BaseCompareTaskParameters
from sandbox.projects.yabs.qa.utils.resource import sync_resource
from sandbox.projects import resource_types
from sandbox.common.types.misc import NotExists

from sandbox.projects.common.yabs.server.tracing import TRACE_WRITER_FACTORY
from sandbox.projects.yabs.sandbox_task_tracing import trace_calls, trace_entry_point
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.generic import new_resource
from sandbox.projects.yabs.sandbox_task_tracing.wrappers.sandbox.sdk2 import make_resource_ready


DEFAULT_MAX_DIFF_PERCENT = 0.1
THE_SAME_BASES_TAGS = 'CHECKED_THE_SAME_BASES'

_PRE_RESULT_KEY = 'pre_task_id'
_TEST_RESULT_KEY = 'test_task_id'
_MAX_DIFF_PERCENT_KEY = 'max_diff_percent'
_COMPARE_SAME_BASES = 'compare_same_bases'


class YabsServerDBSizeCmp(BaseCompareTask):
    name = 'YABS_SERVER_DB_SIZE_CMP'

    class Parameters(BaseCompareTaskParameters):
        cmp_input_parameters = create_input_parameters(default_max_diff_percent=DEFAULT_MAX_DIFF_PERCENT)
        use_chkdb_without_hash = sdk2.parameters.Bool('Use chkdb without hash', default=False)
        ignored_lines = sdk2.parameters.JSON(
            'Ignored lines (feature under development)',
            default=[],
            description='List of regular expressions ignored in chkdb output (in `diff -I` syntax).'
        )
        shard_key = sdk2.parameters.String('Shard key')

    @trace_entry_point(writer_factory=TRACE_WRITER_FACTORY)
    def on_execute(self):
        baseline_ctx = self._get_compared_task_context(_PRE_RESULT_KEY)
        test_ctx = self._get_compared_task_context(_TEST_RESULT_KEY)
        max_diff_percent = getattr(self.Parameters, _MAX_DIFF_PERCENT_KEY)
        compare_same_bases = getattr(self.Parameters, _COMPARE_SAME_BASES)

        has_diff, report = self.compare_contexts(baseline_ctx, test_ctx, max_diff_percent, compare_same_bases)
        if self.Parameters.check_input_param_for_equality:
            self.check_for_equality(baseline_ctx, test_ctx, self.Parameters.check_input_param_for_equality)

        self.Context.has_diff = has_diff
        self.set_info(report)
        fu.write_file('report.txt', report)
        new_resource(resource_types.PLAIN_TEXT, self, 'Yabs server B2B db_size test diff', 'report.txt', ttl=42)

    @trace_calls
    def compare_contexts(self, pre_ctx, test_ctx, max_diff_percent, compare_same_bases):
        ctx_key = CHKDB_WITHOUT_HASH_CTX_KEY if self.Parameters.use_chkdb_without_hash else CHKDB_CTX_KEY
        joint_bases = _join_bases(pre_ctx, test_ctx, compare_same_bases)
        has_diff, report = compare_db_data(
            _get_db_sizes_dict(pre_ctx, joint_bases),
            _get_db_sizes_dict(test_ctx, joint_bases),
            getattr(pre_ctx, ctx_key),
            getattr(test_ctx, ctx_key),
            max_diff_percent,
            sync_resource
        )
        fu.write_file('diff_resource.txt', report)
        diff_resource = new_resource(
            YabsDbSmartMetaReport,
            task=self,
            description='Smart meta report resource',
            path='diff_resource.txt',
            test_name=self.Parameters.test_name,
            arcanum_review_id=self.Context.arcanum_review_id if self.Context.arcanum_review_id is not NotExists else self.Parameters.arcanum_review_id,
            shard_key=self.Parameters.shard_key,
            has_diff=has_diff
        )
        make_resource_ready(diff_resource)
        return has_diff, report

    @property
    def diff_resource_type(self):
        return 'YABS_DB_SMART_META_REPORT'

    @property
    def diff_resource_search_attributes(self):
        return {'shard_key': self.Parameters.shard_key}

    def check_for_equality(self, pre_ctx, test_ctx, param_name):
        pre_bases_set = set(getattr(pre_ctx, DBResourceList.name))
        test_bases_set = set(getattr(test_ctx, DBResourceList.name))
        if pre_bases_set == test_bases_set:
            self.Parameters.tags.append(THE_SAME_BASES_TAGS)


def _join_bases(pre_ctx, test_ctx, compare_same_bases):
    pre_keys = set(getattr(pre_ctx, DB_SIZE_CTX_KEY).keys())
    test_keys = set(getattr(test_ctx, DB_SIZE_CTX_KEY).keys())
    return pre_keys & test_keys if compare_same_bases else pre_keys | test_keys


def _get_db_sizes_dict(ctx, joint_bases):
    db_size = getattr(ctx, DB_SIZE_CTX_KEY)
    logging.info(db_size)
    return {k: float(v) for k, v in db_size.iteritems() if k in joint_bases}


@trace_calls
def compare_db_data(baseline_sizes, test_sizes, baseline_chkdb, test_chkdb, max_diff_percent, sync_resource, chkdb_diff_is_diff=False):
    has_chkdb_diff, chkdb_report = compare_chkdb(baseline_chkdb, test_chkdb, sync_resource)
    has_size_diff, size_report = compare_db_sizes(baseline_sizes, test_sizes, max_diff_percent)

    has_diff = has_size_diff
    if chkdb_diff_is_diff:
        has_diff = has_diff or has_chkdb_diff

    return has_diff, size_report + "\n\n" + chkdb_report
