import shutil
import logging
import os.path

import sandbox.common.types.task as ctt

from sandbox.common import rest

from sandbox.sandboxsdk.parameters import ResourceSelector, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import get_logs_folder, make_folder

from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types

from sandbox.projects.common.utils import get_or_default


class FastMode(SandboxBoolParameter):
    name = 'fast_mode'
    description = 'Fast mode (try to use cache, only current state - no capacity estimates)'
    default_value = False


class EstimatorBinary(ResourceSelector):
    name = 'estimator_binary'
    description = 'yabs-server-capacity-estimate binary'
    default_value = 253928178  # FIXME switch to last released of specific type?
    resource_type = resource_types.YABS_SERVER_PREDICT_LOAD_EXECUTABLE


class YabsServerRunCapacityEstimate(SandboxTask):  # pylint: disable=R0904
    execution_space = 10 * 1024

    SE_TAGS = {'limit': 1}

    type = 'YABS_SERVER_RUN_CAPACITY_ESTIMATE'
    cores = 1

    input_parameters = [EstimatorBinary, FastMode]

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.se_tag:
            sem_name = "{}/{}".format(self.type, self.se_tag)
            self.semaphores(ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=sem_name, capacity=self.SE_TAGS[self.se_tag])
                ]
            ))

    def on_execute(self):
        estimator_path = self.sync_resource(get_or_default(self.ctx, EstimatorBinary))
        cache_path = os.path.abspath("yabs-server-predict-load-cache.pickle")

        try:
            cache_res_data = rest.Client().resource.read(
                limit=10,
                order='-id',
                type=resource_types.YABS_SERVER_CONFIGURATOR_CACHE.name,
                state='READY',
            )
            cache_res_path = self.sync_resource(cache_res_data['items'][0]['id'])
        except Exception:
            logging.exception("Failed to get cache resource, continuing with empty cache", 0)
            cache_mtime = None
        else:
            shutil.copyfile(cache_res_path, cache_path)
            cache_mtime = os.stat(cache_res_path).st_mtime

        debug_plot_path = os.path.join(os.path.abspath(get_logs_folder()), 'debug_plots.pdf')

        heatmaps_path = make_folder('heatmaps')

        cmdline = [
            estimator_path,
            '--debug-plot', debug_plot_path,
            '--autodetect-closed',
            '--heatmap-prefix', os.path.join(heatmaps_path, 'heatmap'),
            '--do-not-cache-graphite',
            '--cache-path', cache_path,
            '--no-fetch-cache',
            '--solomon',
        ]

        if get_or_default(self.ctx, FastMode):
            cmdline += [
                '--sample-len', str(10),
                '--hosts-cache-ttl', str(7200),
            ]
        else:
            cmdline += [
                '--sample-len', str(60),
                '--hosts-cache-ttl', str(3600),
                '--actual-capacity',
                '--installed-capacity',
            ]

        res = self.create_resource(
            description="YaBS frontend heatmaps",
            resource_path=heatmaps_path,
            resource_type=resource_types.YABS_FRONTEND_HEATMAP,
            arch='any'
        )

        try:
            run_process(cmdline, log_prefix='estimator')
        finally:
            # Steps after heatmap may fail
            if os.listdir(heatmaps_path):
                self.mark_resource_ready(res.id)

            if os.stat(cache_path).st_mtime != cache_mtime:
                cache_res = self.create_resource(
                    description="yabs-server configurator cache by %s" % self.type,
                    resource_path=cache_path,
                    resource_type=resource_types.YABS_SERVER_CONFIGURATOR_CACHE,
                    arch='any'
                )
                self.mark_resource_ready(cache_res.id)


__Task__ = YabsServerRunCapacityEstimate
