import json
import logging
from datetime import datetime, timedelta
from operator import itemgetter
from os.path import join as join_path, realpath, dirname

from sandbox.common import telegram
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
from sandbox import sdk2


class YabsServerToolAntispamTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        yql_token_vault_name = sdk2.parameters.String("YQL Token vault name", default="yql_token", required=True)
        db = sdk2.parameters.String("Yt Cluster", default="hahn", required=True)
        min_filter_ratio = sdk2.parameters.Integer("Min filter ratio", default=1000, required=True)
        max_shows_count = sdk2.parameters.Integer("Max shows count", default=10, required=True)
        process_delay = sdk2.parameters.Integer("Process delay", default=60, required=True)
        telegram_token_vault_name = sdk2.parameters.String("TelegramBot Token vault name", default="telegram_token", required=True)
        chat_id = sdk2.parameters.Integer("Char ID", default=-1001104290235, required=True)

    def GetProcessTime(self):
        frmt = '%Y-%m-%dT%H:%M:%S'

        start_process_time = (datetime.now() - timedelta(minutes=self.Parameters.process_delay)).replace(minute=0, second=0)
        end_process_time = start_process_time + timedelta(hours=1)

        return start_process_time.strftime(frmt), end_process_time.strftime(frmt)

    def GetRequest(self):
        return '''
            pragma inferscheme;

            select
                C.orderid,
                F.f / C.s as filter_ratio,
                C.s as shows
            from
            (
                select
                    orderid,
                    count(*) as s
                from range(
                    [//home/logfeller/logs/bs-chevent-log/stream/5min],
                    [{start_process_time}],
                    [{end_process_time}]
                )
                where
                    countertype == '1'
                group by
                    orderid
            ) as C
            inner join
            (
                select
                    orderid,
                    count(*) as f
                from range(
                    [//home/logfeller/logs/bs-filter-log/stream/5min],
                    [{start_process_time}],
                    [{end_process_time}]
                )
                where
                    object in (
                        'ORDER',
                        'BANNER',
                        'PHRASE',
                        'PLACED_BANNER',
                        'CREATIVE',
                    )
                group by
                    orderid
            ) as F
            using(orderid)
            where F.f / C.s > {ratio}
                and C.s < {shows};
        '''.format(
            start_process_time=self.start_process_time,
            end_process_time=self.end_process_time,
            ratio=self.Parameters.min_filter_ratio,
            shows=self.Parameters.max_shows_count
        )

    def on_execute(self):
        self.yql_token = sdk2.task.Vault.data(self.author, self.Parameters.yql_token_vault_name)
        self.telegram_token = sdk2.task.Vault.data(self.author, self.Parameters.telegram_token_vault_name)

        try:
            self._on_execute_unsafe()
        except Exception as e:
            bot = telegram.TelegramBot(self.telegram_token)
            bot.send_message(self.Parameters.chat_id, "Exception: {what}".format(what=str(e)))

    def _on_execute_unsafe(self):
        logging.info("Started!")

        self.start_process_time, self.end_process_time = self.GetProcessTime()
        logging.info("Gonna run on {frm} - {to}".format(frm=self.start_process_time, to=self.end_process_time))

        self.request = self.GetRequest()
        logging.info("Prepared request " + self.request)

        logging.info("Creating virtualenv...")
        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('yql', venv=venv, version='1.2.8').prepare()

            script_path = join_path(dirname(realpath(__file__)), 'yql_runner.py')

            out, err = run_process(
                [
                    venv.executable,
                    script_path,
                    '--query="{}"'.format(self.request),
                    '--token={}'.format(self.yql_token),
                    '--db={}'.format(self.Parameters.db),
                ],
                log_prefix='yql_runner',
                shell=True,
                check=True,
                wait=True,
                outs_to_pipe=True,
            ).communicate()

        out, err = out.decode(), err.decode()

        logging.info("OUT: {}".format(out))
        logging.info("ERR: {}".format(err))

        orders = json.loads(out)
        orders.sort(key=itemgetter('filter_ratio'))

        if len(orders) > 0:
            bot = telegram.TelegramBot(self.telegram_token)
            bot.send_message(self.Parameters.chat_id, "Found {count} bad orders for hour {hour}".format(count=len(orders), hour=self.start_process_time))
            for order in orders:
                bot.send_message(self.Parameters.chat_id, json.dumps(order, indent=2, sort_keys=True))

        logging.info("Done!")
