from datetime import datetime, timedelta
from operator import itemgetter
from os.path import join as join_path, realpath, dirname
from sandbox.common import telegram
from sandbox.sandboxsdk.process import run_process
import json
import logging
from sandbox import sdk2


class YabsServerToolShowProbabilityArbiterTask(sdk2.Task):
    privileged = True

    # class Requirements(sdk2.Task.Requirements):
    #     privileged = True

    class Parameters(sdk2.Task.Parameters):
        yql_token_vault_name = sdk2.parameters.String("YQL Token vault name", default="yql_token", required=True)
        db = sdk2.parameters.String("Yt Cluster", default="hahn", required=True)
        telegram_token_vault_name = sdk2.parameters.String("TelegramBot Token vault name", default="telegram_token", required=True)
        chat_id = sdk2.parameters.Integer("Chat ID", default=-208925150, required=True)

    def GetProcessTime(self):
        frmt = '%Y-%m-%d'
        start_time = datetime.now() - timedelta(days=1)
        return start_time.strftime(frmt)

    def GetUpdateOfShowProbability(self, pages):
        MIN_SHOW_PROBABILITY = 500000
        MAX_SHOW_PROBABILITY_0 = 0
        MAX_SHOW_PROBABILITY_100 = 1000000
        DECREASE_QUERY = '''update PageDSP
                          set ShowProbability = (case
                                                 when (ShowProbability in ({max_show_probability_0}, {max_show_probability_100}) and {max_show_probability_100} * {coef} >= {min_show_probability}) or
                                                      (ShowProbability > {min_show_probability} and ShowProbability * {coef} >= {min_show_probability})
                                                 then ShowProbability * {coef}
                                                 else {min_show_probability}
                                                 end)
                          where PageID = {page_id} and
                                DSPID != 10;\n"
            '''
        INCREASE_QUERY = '''update PageDSP
                          set ShowProbability = (case
                                                 when ShowProbability != {max_show_probability_0} and
                                                      ShowProbability < {max_show_probability_100} and
                                                      ShowProbability * {coef} <= {max_show_probability_100}
                                                 then ShowProbability * {coef}
                                                 else {max_show_probability_0}
                                                 end)
                          where PageID = {page_id} and
                                DSPID != 10;\n"
            '''
        updates = ""
        pages_with_no_stat = []
        for page in pages:
            if page['EventCostSumVsSSPPriceSum'] <= 0:
                pages_with_no_stat.append(page['PageID'])
            elif page['EventCostSumVsSSPPriceSum'] < 1:
                updates += DECREASE_QUERY.format(coef=page['EventCostSumVsSSPPriceSum'],
                                                 page_id=page['PageID'],
                                                 min_show_probability=MIN_SHOW_PROBABILITY,
                                                 max_show_probability_0=MAX_SHOW_PROBABILITY_0,
                                                 max_show_probability_100=MAX_SHOW_PROBABILITY_100)
            else:
                updates += INCREASE_QUERY.format(coef=page['EventCostSumVsSSPPriceSum'],
                                                 page_id=page['PageID'],
                                                 max_show_probability_0=MAX_SHOW_PROBABILITY_0,
                                                 max_show_probability_100=MAX_SHOW_PROBABILITY_100)

        if len(pages_with_no_stat) > 0:
            updates += "update PageDSP set ShowProbability = {min_show_probability} where PageID in ({page_ids}) and DSPID != 10;\n" \
                       .format(min_show_probability=MIN_SHOW_PROBABILITY, page_ids=",".join(map(str, pages_with_no_stat)))
        return updates

    def GetRequest(self):
        return '''
        SELECT RTBSSP.PageID AS PageID,
               RTBSSP.SSPPriceSum AS SSPPriceSum,
               CHEVENT.EventCostSum AS EventCostSum,
               CASE
               WHEN CHEVENT.EventCostSum IS NOT NULL AND RTBSSP.SSPPriceSum IS NOT NULL AND RTBSSP.SSPPriceSum not IN (0, '0')
               THEN cast(CHEVENT.EventCostSum AS Double)/cast(RTBSSP.SSPPriceSum AS Double)
               ELSE 0
               END AS EventCostSumVsSSPPriceSum
        FROM
             (SELECT RTB.pageid AS PageID,
                     SSP.SSPPriceSum AS SSPPriceSum
             FROM
                  (SELECT DISTINCT pageid
                   FROM [logs/bs-rtb-log/1d/{start_process_time}]
                   WHERE sspid NOT IN ('0', '1000', '7944603')) AS RTB
             LEFT JOIN
                  (SELECT pageid,
                          sum_if(cast(sspprice AS Double), sspprice IS NOT NULL and sspprice != '') * 1.18 AS SSPPriceSum
                  FROM [logs/bs-ssp-log/1d/{start_process_time}]
                  GROUP BY pageid) AS SSP
             ON RTB.pageid = SSP.pageid) AS RTBSSP
        LEFT JOIN
             (SELECT pageid AS PageID,
              sum_if(cast(eventcost AS UInt64), eventcost IS NOT NULL AND eventcost != '') * 30 AS EventCostSum
              FROM [logs/bs-chevent-log/1d/{start_process_time}]
              WHERE String::Contains(options, 'dsp') AND
                    (fraudbits IS NULL OR fraudbits IN ('0', '')) AND
                    countertype = '2'
              GROUP BY pageid) AS CHEVENT
        ON RTBSSP.PageID = CHEVENT.PageID;
        '''.format(
            start_process_time=self.start_process_time
        )

    def on_execute(self):
        logging.info("Started!")

        self.yql_token = sdk2.task.Vault.data(self.author, self.Parameters.yql_token_vault_name)
        self.telegram_token = sdk2.task.Vault.data(self.author, self.Parameters.telegram_token_vault_name)
        bot = telegram.TelegramBot(self.telegram_token)

        self.start_process_time = self.GetProcessTime()
        logging.info("Gonna run request for {request_date} date".format(request_date=self.start_process_time))

        self.request = self.GetRequest()
        logging.info("Prepared request " + self.request)

        try:
            logging.info("Installing yandex-yt-python and yandex-yql-python packages...")
            run_process("sudo apt-get update && sudo apt-get install yandex-yt-python "
                        "yandex-yql-python -q0 --assume-yes --force-yes -o "
                        "Dpkg::Options::='--force-confdef'", shell=True, log_prefix='install_yt_yql')
            script_path = join_path(dirname(realpath(__file__)), 'yql_runner.py')

            logging.info("Running yql_runner.py script...")
            out, err = run_process(
                [
                    "python",
                    script_path,
                    "--query=\"{}\"".format(self.request),
                    "--token={}".format(self.yql_token),
                    "--db={}".format(self.Parameters.db),
                ],
                shell=True,
                check=False,
                wait=False,
                outs_to_pipe=True
            ).communicate()

            out, err = out.decode(), err.decode()

            logging.info("OUT: {}".format(out))
            logging.info("ERR: {}".format(err))

            full_yql_answer = json.loads(out)
            pages = full_yql_answer["result"]
            pages.sort(key=itemgetter('EventCostSumVsSSPPriceSum'))
        except Exception, e:
            bot.send_message(self.Parameters.chat_id, "date: {date}, error: {error}".format(date=self.start_process_time, error=e))
            raise Exception(e)

        bot.send_message(self.Parameters.chat_id, "!" * 100)
        bot.send_message(self.Parameters.chat_id, "({}) Public link for the request is: {} ".format(self.start_process_time, full_yql_answer["public_link"]))

        pages_with_no_clicks = filter(lambda x: x['EventCostSum'] is None, pages)
        pages_with_income_less_then_spends = filter(lambda x: x['EventCostSumVsSSPPriceSum'] < 1, pages)

        logging.info("Updates for PageIDs where no clicks:\n{updates}".format(updates=self.GetUpdateOfShowProbability(pages_with_no_clicks)))
        logging.info("Updates for PageIDs where income less then spends:\n{updates}".format(updates=self.GetUpdateOfShowProbability(pages_with_income_less_then_spends)))

        message_no_clicks = "Count of pages where we have no clicks: {}\n(these are not all pages. Check the SB job for more information.)" \
                            .format(len(pages_with_no_clicks))
        message_income_less_then_spends = "Count of pages where our income less then spends: {}\n(these are not all pages. Check the SB job for more information.)" \
                                          .format(len(pages_with_income_less_then_spends))

        bot.send_message(self.Parameters.chat_id, message_no_clicks)
        bot.send_message(self.Parameters.chat_id, message_income_less_then_spends)

        logging.info("Done!")
