# -*- coding: utf-8 -*-

from sandbox.projects.common.utils import get_or_default
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import environments
import sandbox.sandboxsdk.parameters as sbparam
from sandbox.sandboxsdk.process import run_process

import json
import requests
from distutils.version import LooseVersion
from os.path import join as join_path, realpath, dirname

YT_TOKEN_VAULT_NAME = "robot-yabs-yt-backup-token"
YT_DIST = "http://dist.yandex.ru/"
YT_REPO = "common"


def get_yt_pkg_stable_ver(package, default=None):
    pip_deb_packages = {
        "yandex-yt": "yandex-yt-python",
        "yandex-yt-transfer-manager-client": "yandex-yt-transfer-manager-client",
        "yandex-yt-tools": "yandex-yt-python-tools",
    }

    try:
        params = {
            "repo": YT_REPO,
            "pkg": pip_deb_packages[package],
        }
        resp = requests.get("{dist_url}/find/".format(dist_url=YT_DIST), params=params, timeout=2)
        version = "0"
        if resp.status_code == 200:
            for i in resp.json()["result"]:
                if i["environment"] == "stable":
                    if LooseVersion(i["version"]) > LooseVersion(version):
                        version = i["version"]

            if version != "0":
                version_main, version_post = version.split("-")
                return "{package}=={ver_main}.post{ver_post}".format(package=package, ver_main=version_main, ver_post=version_post)
    except Exception:
        return "{package}=={default}".format(package=package, default=default)
    return "{package}=={default}".format(package=package, default=default)


class SrcCluster(sbparam.SandboxStringParameter):
    name = "src_cluster"
    description = "Source YT cluster"


class DstCluster(sbparam.SandboxStringParameter):
    name = "dst_cluster"
    description = "Destination YT cluster"


class OriginCluster(sbparam.SandboxStringParameter):
    name = "origin_cluster"
    description = "Origin backups YT cluster"


class TablesList(sbparam.ListRepeater, sbparam.SandboxStringParameter):
    name = "tables_list"
    description = "Dynamic tables list"
    required = False


class TablesDict(sbparam.DictRepeater, sbparam.SandboxStringParameter):
    name = "tables_dict"
    description = "Input-output tables patterns dict"
    required = False


class BackupDate(sbparam.SandboxStringParameter):
    name = "backup_date"
    description = "Backup date (yyyy-mm-dd)"
    required = False


class MainBlock(sbparam.SandboxInfoParameter):
    name = "main_block"
    description = 'Main params'


class SpecificBlock(sbparam.SandboxInfoParameter):
    name = "specific_block"
    description = 'Specific params. Don\'t change if you not sure'


class DumpQuery(sbparam.SandboxStringParameter):
    name = "dump_query"
    description = "SQL-like predicate to select rows from src tables"
    required = False


class EraseQuery(sbparam.SandboxStringParameter):
    name = "erase_query"
    description = "SQL-like predicate to erase rows from dst table before restore"
    required = False


class TTL(sbparam.SandboxIntegerParameter):
    name = "ttl"
    description = "TTL in days for backup tables"
    required = False


class DeleteTmp(sbparam.SandboxBoolParameter):
    name = 'delete_tmp'
    description = 'Delete tmp dumps'
    default_value = True


class OverwriteDstTable(sbparam.SandboxBoolParameter):
    name = 'overwrite_dst_table'
    description = 'Overwrite dst table if it exist'
    default_value = False
    sub_fields = {'false': [EraseQuery.name]}


class OverwriteBackup(sbparam.SandboxBoolParameter):
    name = 'overwrite_backup'
    description = 'Overwrite backup if it exist'
    default_value = False


class ForceCompaction(sbparam.SandboxBoolParameter):
    name = 'force_compaction'
    description = 'Force compaction after restore'
    default_value = False


class Erasure(sbparam.SandboxBoolParameter):
    name = 'erasure'
    description = 'Convert backup to erasure'
    default_value = True


class QueueName(sbparam.SandboxStringParameter):
    name = "queue_name"
    description = "Queue name for TM"
    required = False


class UseTransferQueue(sbparam.SandboxBoolParameter):
    name = 'use_transfer_queue'
    description = 'Use special queue for TM'
    default_value = False
    sub_fields = {'true': [QueueName.name]}


class CustomPool(sbparam.SandboxStringParameter):
    name = "custom_tm_pool"
    description = "Special pool name for TM"
    required = False


class UseCustomPool(sbparam.SandboxBoolParameter):
    name = 'use_custom_pool'
    description = 'Use special pool for TM (sic!)'
    default_value = False
    sub_fields = {'true': [CustomPool.name]}


class UseBackbone(sbparam.SandboxBoolParameter):
    name = 'use_backbone'
    description = 'Use backbone for transfer'
    default_value = False


class TablesListInput(sbparam.SandboxStringParameter):
    name = "tables_list_input"
    multiline = True
    description = 'List of dynamic tables'


class UseTablesListInput(sbparam.SandboxBoolParameter):
    name = 'use_tables_list_input'
    description = 'Tables list in one field (hack for move)'
    default_value = False
    sub_fields = {'true': [TablesListInput.name]}


class UseSSDBlobs(sbparam.SandboxBoolParameter):
    name = 'use_ssd_blobs'
    description = 'Add ssd_blobs to restored table'
    default_value = False


class CopyAttributes(sbparam.ListRepeater, sbparam.SandboxStringParameter):
    name = "copy_attributes"
    description = "List of attributes to save/copy/restore"
    required = False


class JobCount(sbparam.SandboxIntegerParameter):
    name = "job_count"
    description = "job_count for DumpRestoreClient"
    required = False


class UserSlots(sbparam.SandboxIntegerParameter):
    name = "user_slots"
    description = "user_slots for DumpRestoreClient"
    required = False


class MaxFailedJobCount(sbparam.SandboxIntegerParameter):
    name = "max_failed_job_count"
    description = "max_failed_job_count for DumpRestoreClient"
    required = False


class JobTimeLimit(sbparam.SandboxIntegerParameter):
    name = "job_time_limit"
    description = "job_time_limit for DumpRestoreClient"
    required = False


class YtTokenOwner(sbparam.SandboxStringParameter):
    name = "yt_token_owner"
    description = "User or group that owns yt_token_vault"
    default = "ADVQUALITY"


class YtTokenVault(sbparam.SandboxStringParameter):
    name = "yt_token_vault"
    description = name
    default = YT_TOKEN_VAULT_NAME


class Mode(sbparam.SandboxRadioParameter):
    name = "mode"
    description = "Work mode"
    choices = [
        ("Dynamic -> Static (advanced_backup)", "advanced_backup"),
        ("Static -> Dynamic (advanced_restore)", "advanced_restore"),
        ("Dynamic -> Dynamic (clone)", "clone"),
        ("Backup", "backup"),
        ("Restore", "restore"),
    ]
    sub_fields = {
        'backup': [
            TablesList.name,
            DumpQuery.name,
            TTL.name,
            Erasure.name,
            OverwriteBackup.name,
        ],
        'advanced_backup': [
            TablesDict.name,
            DumpQuery.name,
            TTL.name,
            Erasure.name,
            OverwriteBackup.name,
        ],
        'clone': [
            TablesDict.name,
            DumpQuery.name,
            OverwriteDstTable.name,
            EraseQuery.name,
            ForceCompaction.name,
            UseSSDBlobs.name,
        ],
        'restore': [
            TablesList.name,
            OriginCluster.name,
            BackupDate.name,
            OverwriteDstTable.name,
            EraseQuery.name,
            ForceCompaction.name,
            UseSSDBlobs.name,
        ],
        'advanced_restore': [
            TablesDict.name,
            DumpQuery.name,
            OverwriteDstTable.name,
            EraseQuery.name,
            ForceCompaction.name,
            UseSSDBlobs.name,
        ],
    }


class YabsYTBackupRestore19(SandboxTask):
    type = 'YABS_YT_BACKUP_RESTORE_19'

    input_parameters = [
        MainBlock,
        Mode,
        SrcCluster,
        DstCluster,
        TablesList,
        TablesDict,
        OriginCluster,
        BackupDate,
        TTL,
        Erasure,
        DumpQuery,
        OverwriteBackup,
        OverwriteDstTable,
        EraseQuery,
        SpecificBlock,
        DeleteTmp,
        ForceCompaction,
        UseTransferQueue,
        QueueName,
        UseCustomPool,
        CustomPool,
        UseBackbone,
        UseTablesListInput,
        TablesListInput,
        UseSSDBlobs,
        CopyAttributes,
        JobCount,
        UserSlots,
        MaxFailedJobCount,
        JobTimeLimit,
        YtTokenOwner,
        YtTokenVault,
    ]

    def on_execute(self):
        params = self.ctx
        yt_token = self.get_vault_data(
            get_or_default(params, YtTokenOwner) or YtTokenOwner.default,
            get_or_default(params, YtTokenVault) or YtTokenVault.default,
        )

        params.update(
            yt_token=str(yt_token),
            task_id=self.id,
        )

        self.descr = "{} from {} to {} : ".format(
            params[Mode.name],
            params[SrcCluster.name],
            params[DstCluster.name],
        )

        if params[UseTablesListInput.name]:
            self.descr += "{}".format(params[TablesListInput.name].replace('\n', ', '))
        else:
            if params[Mode.name] in ('backup', 'restore'):
                self.descr += "{}".format(list(params[TablesList.name]))
            else:
                self.descr += "{}".format(dict(params[TablesDict.name]))

        if params[Mode.name] == 'restore':
            self.descr += " from backup {}/{}".format(
                params[OriginCluster.name],
                params[BackupDate.name],
            )

        if params.get(DumpQuery.name):
            self.descr += ' by DumpQuery "{}"'.format(params[DumpQuery.name])

        if params.get(EraseQuery.name):
            self.descr += ' by EraseQuery "{}"'.format(params[EraseQuery.name])

        with environments.VirtualEnvironment(use_system=True) as venv:
            venv.pip("pip==9.0.1 setuptools==33.1.1")  # hardcode ver because of https://github.com/pypa/setuptools/issues/942
            requirements = (
                # get_yt_pkg_stable_ver("yandex-yt", "0.7.34.post0"),
                "yandex-yt=={ver}".format(ver="0.8.38a1"),
                get_yt_pkg_stable_ver("yandex-yt-transfer-manager-client", "0.0.26.post0"),
                # get_yt_pkg_stable_ver("yandex-yt-tools", "0.2.108.post0"),
                "yandex-yt-tools=={ver}".format(ver="0.2.128.post0"),
                "yandex-yt-yson-bindings-skynet=={ver}".format(ver="0.3.7.post1"),
                "yandex-yt-driver-rpc-bindings",
            )
            venv._VirtualEnvironment__run_cmd(
                '{python} -us {pip} install -i {pypi} -vvv {req}'.format(
                    python=venv.executable,
                    pip=venv._VirtualEnvironment__pip_wrapper,
                    pypi='https://pypi.yandex-team.ru/simple/',
                    req=" ".join(requirements),
                ), log_prefix='venv_yt'
            )
            run_process(
                [venv.executable, join_path(dirname(realpath(__file__)), 'yt_runner.py'), json.dumps(params)],
                log_prefix='yt_runner'
            )


__Task__ = YabsYTBackupRestore19
