# -*- coding: utf-8 -*-

import time
import requests
import hashlib

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sdk2 import environments
import sandbox.sandboxsdk.parameters as sbparam

from sandbox.projects.common.yabs import ytstat
from sandbox.projects.common.yabs import parameters
from sandbox.projects.common.yabs.app_options import AppOptions

from sandbox.sdk2 import yav

LOG_FILE = "collector.log"


class CollectorName(sbparam.SandboxStringParameter):
    name = "collector_name"
    description = name


class ResourceID(sbparam.SandboxIntegerParameter):
    name = "resource_id"
    description = name


class ReleaseType(sbparam.SandboxStringParameter):
    name = "release_type"
    description = name
    required = True
    default_value = "prod"
    choices = [
        ("Production", "prod"),
        ("Preproduction", "preprod"),
    ]


class Auditor(object):
    TASK_ID_KEY = 'sandbox_task_id'
    RES_NAME_KEY = 'res_type'
    ARCH_NAME_KEY = 'arch_md5'
    BINS_NAME_KEY = 'bins_md5'
    AUDIT_HANDLERS = [
        'http://bsint01e.yabs.yandex.ru:9999/sbyt-audit',
        'http://bsint01f.yabs.yandex.ru:9999/sbyt-audit',
        'http://bsint01i.yandex.ru:9999/sbyt-audit'
    ]
    OK = 'OK'
    FAIL = 'FAIL'

    def __init__(self, task, logger, col_info):
        self._task = task
        self._logger = logger
        self._col_info = col_info

    def _get_md5(self, filename):
        hash_md5 = hashlib.md5()
        with open(filename, 'rb') as bf:
            for chunk in iter(lambda: bf.read(4096), b""):
                hash_md5.update(chunk)
        return hash_md5.hexdigest()

    def _build_audit_request(self):
        result = {}
        result[self.TASK_ID_KEY] = self._task.id
        result[self.RES_NAME_KEY] = self._task.ctx['collector_name']
        result[self.ARCH_NAME_KEY] = self._col_info['resource_md5']
        result[self.BINS_NAME_KEY] = {}
        fname = self._col_info['binary']
        result[self.BINS_NAME_KEY][fname] = self._get_md5('./' + fname)
        return result

    def check(self):
        req = self._build_audit_request()
        for handler in self.AUDIT_HANDLERS:
            try:
                resp = requests.post(handler, json=req, timeout=15)
                if resp.ok:
                    return str(resp.text)
            except requests.exceptions.RequestException as e:
                self._logger.warn('audit is unavailable on link error: {}'.format(str(e)))
        return ""


class YabsYTStatCollector(SandboxTask):
    type = 'YABS_YT_STAT_COLLECTOR'
    environment = (
        environments.PipEnvironment("yandex-yt", "0.8.38a1", use_wheel=True),
    )

    input_parameters = list(parameters.CollectorParameters) + [CollectorName, ResourceID, ReleaseType]
    client_tags = ctc.Tag.LXC | ctc.Tag.PORTOD
    cores = 1

    def on_execute(self):
        # Compatibility with old coordinator
        if "yt_proxy" not in self.ctx:
            self.ctx["yt_proxy"] = self.ctx["yt_proxy_url"]
        ytstat.prepare_log_file(self, LOG_FILE)
        with ytstat.stderr_redirect(LOG_FILE):
            self._do_on_execute()

    def _do_on_execute(self):
        ctx = self.ctx
        on_execute_tm = time.time()

        yt_proxy = ctx["yt_proxy"]
        yt_token_secret_id = ctx["yt_token_secret_id"]
        is_development = ctx["is_development"]
        collector_name = ctx["collector_name"]
        resource_id = ctx["resource_id"]
        released = not is_development

        yt_token = None
        if yt_token_secret_id:
            yt_token = yav.Secret(yt_token_secret_id).data()["YT_TOKEN"]

        # Install supervisor and import some modules
        ytc = ytstat.get_yt_client(yt_proxy, yt_token)
        ytstat.install_supervisor(self, ytc, released)

        from supervisor import ylogging
        from supervisor import collector
        logger = ylogging.get_logger()

        # Download application options
        AppOptions.get_app_options_resource(self)
        app_options_file = AppOptions.make_app_options_file(yt_proxy, dev=is_development)
        logger.info("prepare app options file (%s)", app_options_file)

        # Extract env variables
        env_variables = AppOptions.get_env_variables(app_options_file)
        logger.info("env variables extracted")

        # Install
        ytstat.install_resource(self, resource_id)
        logger.info("install_resource(%d)", resource_id)

        # Get info
        sb_api = ytstat.SbApi(self)
        col_info = sb_api.get_collector_info_by_resource_id(resource_id, collector_name)

        # Audit
        if ctx['is_auditable']:
            auditor = Auditor(self, logger, col_info)
            check = auditor.check()
            if check != auditor.OK:
                logger.warn('collector {} has not passed audit'.format(collector_name))
            else:
                logger.info('collector {} has passed audit'.format(collector_name))
        else:
            logger.info('audit disabled')

        new_params = {
            "release_type": ctx.get("release_type"),  # 'get' to be safe if launched by old coordinator
            "version": col_info["version"],
            "group_name": col_info["group_name"],
            "lock_wait_time": ctx["lock_wait_time"],
        }

        # Execute collector
        collector.execute(
            collector.CollectorParams(
                yt_proxy=yt_proxy,
                yt_token=yt_token,
                name=collector_name,
                binary=col_info["binary"],
                log_path=self.ctx["log_path"],
                task_id=self.id,
                app_options_file=app_options_file,
                env_variables=env_variables,
                # TODO Move new_params here when new supervisor deployed
                **{k: v for k, v in new_params.items() if hasattr(collector.CollectorParams, k)}
            ),
            collector.StopwatchTimes(
                task_create=ytstat.get_task_create_time(self.id),
                on_execute=on_execute_tm,
            )
        )


__Task__ = YabsYTStatCollector
