import os
import logging
import datetime

from sandbox import sdk2

from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs import svn

from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common import constants
from sandbox.projects.common.vcs import arc

YNDEX_UPLOADER = 'devtools/codenav/uploader/codenav-uploader'
YNDEXING_TIME = 4 * 60 * 60  # 10 hours


class Yndexer(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl("Arcadia Url")
        patch = sdk2.parameters.String('Patch', multiline=True)
        targets = sdk2.parameters.String('Targets')
        projects_partitions_count = sdk2.parameters.String('Projects partitions count', default=1)
        projects_partition_index = sdk2.parameters.String('Projects partition index', default=0)
        db_password_vault_key = sdk2.parameters.String('Database password Vault key')
        yt_token_vault_key = sdk2.parameters.String('YT token Vault key')
        db_host = sdk2.parameters.String('Database host')
        db_port = sdk2.parameters.Integer('Database port')
        db_name = sdk2.parameters.String('Database name')
        db_user = sdk2.parameters.String('Database user')
        db_extra = sdk2.parameters.String('Database parameters', default='sslmode=require')
        with sdk2.parameters.RadioGroup('Build system') as build_system:
            build_system.values[constants.YA_MAKE_FORCE_BUILD_SYSTEM] = build_system.Value(value=constants.YA_MAKE_FORCE_BUILD_SYSTEM, default=True)
            build_system.values[constants.DISTBUILD_FORCE_BUILD_SYSTEM] = build_system.Value(value=constants.DISTBUILD_FORCE_BUILD_SYSTEM)
            build_system.values[constants.YMAKE_BUILD_SYSTEM] = build_system.Value(value=constants.YMAKE_BUILD_SYSTEM)
        rebuild = sdk2.parameters.Bool('Rebuild', default=False)
        distbuild_pool = sdk2.parameters.String('Distbuild pool')
        yt_cluster = sdk2.parameters.String('Yt cluster')
        yt_root = sdk2.parameters.String('Yt root')
        java_yndexing = sdk2.parameters.Bool('Enable java yndexing')
        go_yndexing = sdk2.parameters.Bool('Enable GO yndexing')
        py3_yndexing = sdk2.parameters.Bool("Enable Python3 yndexing", default=False)
        py_yndexing = sdk2.parameters.Bool("Enable legacy Python2 yndexing", default=True)
        build_doc = sdk2.parameters.Bool('Update build documentation', default=False)
        curfew = sdk2.parameters.Bool('Enable curfew for task( do not run between 10:00 and 22:00 )', default=True)

        use_arc_instead_svn = sdk2.parameters.Bool("Use Arc instead of SVN", default=False)
        with use_arc_instead_svn.value[True]:
            ya_secret = sdk2.parameters.YavSecret('Yav secret to update resources', default='sec-01fmvmxw2jxvc7pvkmqq66tagj')

    def on_execute(self):
        self.logger = logging.getLogger("Yndexer")
        self.mount_point = None

        if self.Parameters.curfew and (10 <= datetime.datetime.now().hour < 22):
            raise Exception('Please do not run this task between 10:00 and 22:00')

        if not self.Parameters.use_arc_instead_svn:
            self.logger.info("Using svn")
            source_root = svn.Arcadia.get_arcadia_src_dir(self.Parameters.arcadia_url)
            revision = svn.Arcadia.get_revision(source_root)
        else:
            self.logger.info("Using arc")
            if self.Parameters.build_doc:
                raise Exception("Can build docs on arc, disable use_arc_instead_svn")

            arc_token = self.Parameters.ya_secret.data()[self.Parameters.ya_secret.default_key or "ARC_TOKEN"]
            arc_ = arc.Arc(arc_oauth_token=arc_token)
            del arc_token
            arcadia_url = self.Parameters.arcadia_url
            # TODO: Parse arcadia_url
            if arcadia_url.startswith("arcadia-arc:/#"):
                self.logger.debug("Remove `arcadia-arc:/#` prefix")
                arcadia_url = arcadia_url.replace("arcadia-arc:/#", "")

            self.logger.debug("Try to use arcadia_url: `%s` (original: `%s`)", arcadia_url, self.Parameters.arcadia_url)
            self.mount_point = arc_.mount_path(None, changeset=arcadia_url, fetch_all=False)
            source_root = self.mount_point._work_path

            full_revision = "\n".join(arc_.describe(source_root, svn=True))
            revision = full_revision.split("-")[0].replace("r", "")
            self.logger.debug("Describe: %s", full_revision)
            self.logger.debug("svn revision: %s", revision)

        if self.Parameters.patch:
            self.logger.info("Applying path %s", self.Parameters.patch)
            svn.Arcadia.apply_patch(source_root, self.Parameters.patch, str(self.log_path().absolute()))

        merged_yndex_root = str(self.log_path('merged_yndex').absolute())
        yndex_root = str(self.log_path('yndex').absolute())
        bin_root = str(self.log_path('bin').absolute())

        targets = self.Parameters.targets.split(';')
        build_system = self.Parameters.build_system
        distbuild_pool = self.Parameters.distbuild_pool

        self.logger.debug(
            'Run task with params: ' + '\n'.join(
                'source root is %s, '
                'revision is %s, '
                'merged yndex root %s, '
                'yndex root %s, '
                'bin root %s, '
                'targets %s, '
                'build system %s'),
            source_root, revision, merged_yndex_root, yndex_root, bin_root, targets, build_system)

        if self.Parameters.build_doc:
            self.logger.debug('Updating build documentation: source root is %s, revision is %s', source_root, revision)
            with sdk2.helpers.ProcessLog(self, logger='doc_updater') as pl:
                with open(os.path.join(source_root, 'build', 'docs', 'readme.md'), 'w') as readme_md:
                    sp.Popen(
                        [os.path.join(source_root, 'ya'), 'dump', 'conf-docs'],
                        stdout=readme_md, stderr=pl.stderr, cwd=source_root,
                    ).wait()
                with open(os.path.join(source_root, 'build', 'docs', 'all.md'), 'w') as all_md:
                    sp.Popen(
                        [os.path.join(source_root, 'ya'), 'dump', 'conf-docs', '--dump-all'],
                        stdout=all_md, stderr=pl.stderr, cwd=source_root,
                    ).wait()
                self._commit_docs(source_root, revision)

        self.logger.info("Build codenav uploader via %s", YNDEX_UPLOADER)
        sdk.do_build(
            constants.YMAKE_BUILD_SYSTEM, source_root, [os.path.dirname(YNDEX_UPLOADER)], 'release',
            clear_build=False, results_dir=bin_root, keep_on=True, check_rc=True, no_src_changes=True,
        )

        yt_root = self.Parameters.yt_root + '/' + str(revision)
        yndexing_params = {
            'yt_cluster': self.Parameters.yt_cluster,
            'yt_root': yt_root,
            'java_yndexing': self.Parameters.java_yndexing,
            'py3_yndexing': self.Parameters.py3_yndexing,
            'py_yndexing': self.Parameters.py_yndexing,
        }

        self.logger.debug('Yndexing params: %s', yndexing_params)

        def_flags = {'TRAVERSE_RECURSE_FOR_TESTS': 'yes'}
        if int(self.Parameters.projects_partitions_count) > 1:
            def_flags.update({'RECURSE_PARTITIONS_COUNT': self.Parameters.projects_partitions_count, 'RECURSE_PARTITION_INDEX': self.Parameters.projects_partition_index})
        if self.Parameters.go_yndexing:
            def_flags['GO_VET'] = 'yndexer'

        yt_token = sdk2.Vault.data(self.Parameters.yt_token_vault_key)
        self.logger.info("Building targets %s", targets)
        sdk.do_build(
            build_system, source_root, targets, 'release',
            clear_build=self.Parameters.rebuild, results_dir=yndex_root, keep_on=True, check_rc=False,
            no_src_changes=True, yndexing_params=yndexing_params, download_artifacts=False,
            build_execution_time=YNDEXING_TIME,
            timeout=YNDEXING_TIME,
            coordinators_filter='distbuild-vla',
            def_flags=def_flags,
            distbuild_pool=distbuild_pool,
            env={'YA_TOKEN': yt_token}
        )

        password = sdk2.Vault.data(self.Parameters.db_password_vault_key)
        assert password is not None

        database_params = [
            'dbname={}'.format(self.Parameters.db_name),
            'host={}'.format(self.Parameters.db_host),
            'port={}'.format(self.Parameters.db_port),
            'user={}'.format(self.Parameters.db_user),
            self.Parameters.db_extra,
        ]

        self.logger.info("Transform items from %s", merged_yndex_root)
        with sdk2.helpers.ProcessLog(self, logger='transform') as pl:
            sp.Popen(
                [os.path.join(source_root, 'ya'), 'tool', 'ytyndexer', 'transform', '-y', merged_yndex_root, '-c', self.Parameters.yt_cluster, '-r', yt_root],
                env={'YT_YNDEXER_YT_TOKEN': yt_token}, stdout=pl.stdout, stderr=pl.stderr,
            ).wait()

        self.logger.info("Upload items from %s", merged_yndex_root)
        with sdk2.helpers.ProcessLog(self, logger='uploader') as pl:
            sp.Popen(
                [os.path.join(bin_root, YNDEX_UPLOADER), '--dsn', ' '.join(database_params), '-s', source_root, '-p', merged_yndex_root, '-r', revision],
                stdout=pl.stdout, stderr=pl.stderr, env={'PASSWORD': str(password)}
            ).wait()

    def _commit_docs(self, source_root, revision):
        commit_msg = 'Update build documentation for r{}'.format(revision)
        commit_msg += '\nSKIP_CHECK'

        docs_dir = os.path.join(source_root, 'build', 'docs')
        self.logger.info('Going to make commit to %s with message: %s', docs_dir, commit_msg)

        attempts = 5
        for _ in range(attempts):
            try:
                svn_stdout = svn.Arcadia.commit(docs_dir, commit_msg, 'zomb-sandbox-rw')
                self.logger.info('Documentation committed: {}\nMessage: {}'.format(svn_stdout, commit_msg))
            except svn.SvnError:
                continue
            else:
                return
        self.logger.info('Failed to commit documentation in {} attempts'.format(attempts))
        svn.Arcadia.revert(docs_dir, recursive=True)
        # Keep going

    def on_finish(self, prev_status, status):
        try:
            if hasattr(self, "mount_point") and self.mount_point is not None:
                self.mount_point.unmount()
        except Exception:
            logging.exception("While unmounting %s", self.mount_point._work_path)
