import logging
import os

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox.projects import resource_types
from sandbox.sandboxsdk.process import run_process


class Params(object):
    ''' Input parameters '''
    class YtBackuper(sdk_parameters.ResourceSelector):
        name = 'yt_backuper'
        description = 'Yt-tables backuper'
        resource_type = resource_types.YT_BACKUPER_EXECUTABLE
        required = True

    class YtTable(sdk_parameters.SandboxStringParameter):
        name = 'yt_table_path'
        description = 'Path to YT table which needs to be backuped'
        required = True

    class BackupDirectory(sdk_parameters.SandboxStringParameter):
        name = 'backup_directory_path'
        description = 'Path to destination backup directory'
        default_value = '//tmp'
        required = True

    class DisplacementPeriod(sdk_parameters.SandboxStringParameter):
        name = 'displacement_period'
        description = 'Max numbers of backups to store (0 for disable displacement)'
        default_value = '0'
        required = True

    class YtProxy(sdk_parameters.SandboxStringParameter):
        name = 'yt_proxy'
        description = 'Yt proxy'
        default_value = 'hahn'
        required = True

    Content = [
        YtBackuper,
        YtTable,
        BackupDirectory,
        DisplacementPeriod,
        YtProxy,
    ]


class YtBackup(sdk_task.SandboxTask, Params):
    '''Yt-table backuper.'''

    type = 'YT_BACKUP'
    input_parameters = Params.Content

    environment = [sdk_environments.PipEnvironment('yandex-yt')]

    def on_execute(self):
        yt_backuper = self.sync_resource(self.ctx.get(self.YtBackuper.name))

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = self.get_vault_data(self.owner, 'yt-token')
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        backup_cmd = [yt_backuper,
                      '--backup_dir', self.ctx.get(self.BackupDirectory.name),
                      '--proxy', self.ctx.get(self.YtProxy.name),
                      '--displacement_period', self.ctx.get(self.DisplacementPeriod.name),
                      self.ctx.get(self.YtTable.name)]

        logging.info('Run command: {}'.format(' '.join(backup_cmd)))
        run_process(backup_cmd)
        logging.info('Done')


__Task__ = YtBackup
