import logging

from sandbox.projects.release_machine.helpers.request_helper import RequestHelper

ABC_API = "https://abc-back.yandex-team.ru/api/v4/"


class AbcClient(RequestHelper):
    def __init__(self, token, timeout=30, ssl_verify=False):
        super(AbcClient, self).__init__(ABC_API, timeout, ssl_verify)
        self.__token = token
        self.headers = {
            'content-type': 'application/json',
            'Authorization': 'OAuth ' + self.__token
        }

        self.logger = logging.getLogger(__name__)

    def get_service_info(self, service_id):
        """
        Get all info about service in abc by its id.

        :param service_id: service id
        :return: response
        """
        service_suffix = "services/{}/".format(service_id)
        return self._do_get(service_suffix)

    def get_service_info_by_slug(self, service_slug):
        """
        Get all info about service in abc by its slug.

        :param service_id: service slug
        :return: response
        """
        service_suffix = "services/?slug={}".format(service_slug)
        results = self._do_get(service_suffix)["results"]
        if len(results) > 0:
            return results[0]
        else:
            return None

    def get_people_from_service(self, service_id, role_id=None, role_scope=None, is_robot=None):
        """
        Get all people from service in abc by its id, role (optional), role_scope (optional) and is_robot (optional).

        :return: List of staff logins
        """
        service_suffix = "services/members/?service={}&page_size=1000".format(service_id)
        if role_id is not None:
            service_suffix += "&role={}".format(role_id)
        if role_scope is not None:
            service_suffix += "&role__scope={}".format(role_scope)
        if is_robot is not None:
            service_suffix += "&is_robot={}".format(is_robot)
        response = self._do_get(service_suffix)
        return [human["person"]["login"] for human in response['results']]

    def get_dutywork_from_service(self, service_id):
        """
        Get dutywork from service in abc.

        :return: staff login string
        """
        service_suffix = "services/members/?service={}&role__scope=dutywork".format(service_id)
        response = self._do_get(service_suffix)
        return str(response['results'][0]['person']['login'])

    def get_current_duty_login(self, service_id=None, schedule_slug=None):
        """
        Get current duty login for service.

        :param service_id (optional): service ID
        :param schedule_slug: (optional) schedule__slug, if specified duty for this slug is retrieved
        :return: staff login string
        """
        if service_id:
            suffix = "services/{}/on_duty/".format(service_id)
        else:
            suffix = 'duty/on_duty/'

        if schedule_slug:
            suffix = "{}?schedule__slug={}".format(
                suffix,
                schedule_slug,
            )

        response = self._do_get(suffix)
        if len(response) > 0:
            return str(response[0].get('person', {}).get('login'))
        else:
            return None

    def get_component_id_by_service_slug(self, service_slug):
        service_suffix = "services/?slug={}".format(service_slug)

        response = self._do_get(service_suffix)

        if not isinstance(response, dict):
            return

        response = response.get("results")

        if not response:
            return

        response = response[0]

        return response["id"]

    def get_responsible(self, abc_group):
        if not abc_group.component_id:
            logging.warning("component_id not specified: %s", abc_group)

            logging.info("Trying to get component id by service slug")

            component_id = self.get_component_id_by_service_slug(abc_group.service_name)

            if not component_id:
                logging.error("Unable to retrieve component id")
                return

            logging.info("Component id: %s", component_id)

        else:

            component_id = abc_group.component_id

        try:
            if not abc_group.role_id:
                return self.get_current_duty_login(component_id, abc_group.schedule_slug)
            else:
                people_from_service = self.get_people_from_service(
                    component_id,
                    abc_group.role_id,
                )
                if people_from_service:
                    return people_from_service[0]
        except Exception as e:
            logging.exception("Unable to get responsible from abc: %s", e)
