# -*- coding: utf-8 -*-
import logging

from sandbox.projects.adfox.adfox_ui.util.config import GitConfig
from sandbox.projects.adfox.adfox_ui.util.release_integration import AdfoxReleaseIntegration
from sandbox.sdk2 import parameters, WaitTime, ResourceData
from sandbox.projects.partner.tasks.e2e_tests.misc.resources import PartnerJSONReportResource
from sandbox.sandboxsdk import errors
from sandbox.projects.adfox.adfox_ui.util.duty_tools import dutyTools
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask
from sandbox.projects.adfox.adfox_ui.testpalm.testrunCreate.task import PROJ_ADFOX
import time
import urllib
import json

url_opener = urllib.URLopener()

TRACKER_RUN_TESTS = 'run-hermione-tests'
TRACKER_COLLECT_AND_UPLOAD_REPORT = 'collect-and-upload-report'

REPORT_JSON_FILE = 'report.json'

STAGE_START = 'start'
STAGE_WAIT = 'wait'
STAGE_REPORT = 'report'

TIMEOUT_AFTER_TEST_STARTED = 180
TIMEOUT_GET_NEXT_STATUS = 90
TIMEOUT_TOTAL = 7200

GITLAB_JOB_STATUS_RUNNING = 'running'
GITLAB_JOB_STATUS_FAILED = 'failed'
GITLAB_JOB_STATUS_SUCCESS = 'success'
GITLAB_JOB_STATUS_PENDING = 'pending'
GITLAB_JOB_STATUS_CREATED = 'created'

GITLAB_LABEL_ADFOX_TEST_SUCCESS_FINISH = 'adfox_tests_ok'


class AdfoxUiTest(AnalyzableTask):
    """ Запускает UI тесты в Adfox """

    autotest_stats = []
    adfox_ui_test_stage = None
    autotest_started_at = None

    class Parameters(AnalyzableTask.Parameters):

        gitBranch = parameters.String(
            'Git branch',
            default=None,
            description="Ветка git"
        )

        piFrontendTag = parameters.String(
            'Pi frontend image tag',
            default=None,
            required=True,
            description="Тэг образа фронта"
        )

        piBackendTag = parameters.String(
            'Pi backend image tag',
            default=None,
            required=True,
            description="Тэг образа перл-бэка"
        )

        javaTag = parameters.String(
            'Pi java image tag',
            default=None,
            required=True,
            description="Тэг java-образов"
        )

    def get_adfox_branch(self):
        if not self.Parameters.gitBranch:
            return AdfoxReleaseIntegration.resolve_branch(GitConfig.RELEASE_CANDIDATE_PREFIX)

        return self.Parameters.gitBranch

    def on_execute(self):
        super(AdfoxUiTest, self).on_execute()
        self.adfox_ui_test_stage = self.get_task_stage()
        if not self.adfox_ui_test_stage:
            self.set_task_stage(STAGE_START)
        if self.adfox_ui_test_stage == STAGE_START:
            self.track_start(TRACKER_RUN_TESTS)
            job_ids = dutyTools.start_autotests(
                branch=str(self.get_adfox_branch()),
                front_tag=str(self.Parameters.piFrontendTag),
                backend_tag=str(self.Parameters.piBackendTag),
                java_tag=str(self.Parameters.javaTag)
            )
            dutyTools.autotest_notify_start(self)
            self.Context.autotest_job_ids = job_ids
            self.Context.autotest_started_at = time.time()
            self.set_task_stage(STAGE_WAIT)
            raise WaitTime(TIMEOUT_AFTER_TEST_STARTED)
        if (time.time() - self.Context.autotest_started_at) > TIMEOUT_TOTAL:
            raise Exception('Total timeout')
        if self.adfox_ui_test_stage == STAGE_WAIT:
            jobs = dutyTools.get_gitlab_jobs(self.Context.autotest_job_ids)
            for job in jobs:
                task_status = job.get("status")
                if (task_status == GITLAB_JOB_STATUS_RUNNING):
                    raise WaitTime(TIMEOUT_GET_NEXT_STATUS)
                if (task_status == GITLAB_JOB_STATUS_PENDING) or (task_status == GITLAB_JOB_STATUS_CREATED):
                    raise WaitTime(TIMEOUT_GET_NEXT_STATUS)
                if task_status == GITLAB_JOB_STATUS_FAILED:
                    self.Context.has_failed = 1
                    self.Context.save()
            self.track_finish(TRACKER_RUN_TESTS)
            self.set_task_stage(STAGE_REPORT)
        if self.adfox_ui_test_stage == STAGE_REPORT:
            self.track_start(TRACKER_COLLECT_AND_UPLOAD_REPORT)
            jobs = dutyTools.get_gitlab_jobs(self.Context.autotest_job_ids)
            common_report_json = {}
            for job in jobs:
                job_id = job.get('id')
                report_url = dutyTools.get_autotest_report_url(job_id)
                status = 'OK' if job.get('status') == GITLAB_JOB_STATUS_SUCCESS else 'FAIL'
                self.set_info('job {} {} status {} - {}'.format(
                    job_id,
                    job.get('name'),
                    status,
                    '<a href="{}">Report</a>'.format(report_url) if report_url else 'REPORT NOT FOUND'
                ), do_escape=False)
                if report_url:
                    report_json_url = report_url.replace('index.html', 'report.json')
                    logging.info('report_json_url: {}'.format(report_json_url))
                    report_json_local_file = '{}_report.json'.format(job_id)
                    url_opener.retrieve(report_json_url, report_json_local_file)
                    data = json.loads(open(report_json_local_file).read())
                    for key in data.keys():
                        common_report_json[key] = data[key]
                else:
                    self.Context.not_found_report = 1
                    self.Context.save()

            with open(REPORT_JSON_FILE, 'w') as f:
                json.dump(common_report_json, f)

            self._collect_autotest_stats(common_report_json)
            dutyTools.send_autotest_metrics(self.autotest_stats)

            ResourceData(PartnerJSONReportResource(self, 'JSON Report', REPORT_JSON_FILE)).ready()
            self.track_finish(TRACKER_COLLECT_AND_UPLOAD_REPORT)

            if self.Context.not_found_report or self.Context.has_failed:
                dutyTools.autotest_notify_fail(self)
                raise errors.SandboxTaskFailureError('Test failure')
            dutyTools.autotest_notify_success(self)

            if self.Parameters.gitBranch:
                dutyTools.mark_release(self.Parameters.gitBranch, GITLAB_LABEL_ADFOX_TEST_SUCCESS_FINISH)

    def set_task_stage(self, stage):
        logging.info('set_task_stage: {}'.format(stage))
        self.Context.adfox_ui_test_stage = stage
        self.adfox_ui_test_stage = stage
        self.Context.save()

    def get_task_stage(self):
        return self.Context.adfox_ui_test_stage

    def _collect_autotest_stats(self, report):
        for key, value in report.iteritems():
            result = 'ok' if value['status'] == 'success' else value['status']
            duration = 0 if value['duration'] is None else value['duration']

            self._add_autotest_metric(key, 'duration', duration, result)
            if 'meta' in value and 'stats' in value['meta']:
                for meta_key, meta_value in value['meta']['stats'].iteritems():
                    m_value = 0 if meta_value is None else meta_value
                    self._add_autotest_metric(key, meta_key, m_value, result)

    def _add_autotest_metric(self, test_id, test_type, value, result):
        self.autotest_stats.append({
            'project': PROJ_ADFOX,
            'release_id': self.metrics_release_id,
            'external_id': self.metrics_id,
            'run_id': self.run_id,
            'test_id': test_id,
            'metric_type': test_type,
            'metric_value': value,
            'result': result,
        })
