# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2
from sandbox.projects.common.juggler import jclient
from datetime import datetime, timedelta
from ..util.config import JugglerConfig


DEFAULT_JUGGLER_SERVICE = 'adfox-gitlab-backup-validation'


class AdfoxGitlabBackupValidation(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("YAV secrets") as yav_secrets:
            s3_secret = sdk2.parameters.YavSecret(
                'Secret for aws s3',
                default='sec-01d612wkkfbwngeyx0807505xw',
                required=True,
            )

        with sdk2.parameters.Group("Juggler Parameters") as juggler_block:
            juggler_host = sdk2.parameters.String('Juggler host',
                                                  default=JugglerConfig.DEFAULT_JUGGLER_HOST,
                                                  required=True)
            juggler_service = sdk2.parameters.String(
                'Juggler service',
                default=DEFAULT_JUGGLER_SERVICE,
                required=True
            )

    def on_execute(self):
        from sandbox.projects.adfox.adfox_ui.util.mds import MdsS3

        secrets = sdk2.yav.Yav(
            s3=self.Parameters.s3_secret,
        )

        mds = MdsS3(url='http://s3.mds.yandex.net',
                    bucket_name='adfox',
                    access_key=secrets.s3['aws_access_key_id'],
                    secret_key=secrets.s3['aws_secret_access_key'])

        yesterday = datetime.today() - timedelta(days=1)
        logging.info(yesterday)

        date = yesterday.strftime('%Y%m%d')
        res = mds.list('gitlab/{}/'.format(date))

        if 'Contents' not in res:
            self.juggler_notify_error()
            return

        for s3File in res['Contents']:
            logging.info(s3File['Key'])
            logging.info(s3File['Size'])
            if '_gitlab_backup.tar' in s3File['Key'] and int(s3File['Size']) > 1024 * 1024 * 1024:
                self.juggler_notify_ok()
                return

        self.juggler_notify_error()

    def juggler_notify_error(self):
        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            'CRIT',
            'No gitlab backup yesterday. More info: https://wiki.yandex-team.ru/adfox/interfaces/Бэкапы-gitlab/'
        )

    def juggler_notify_ok(self):
        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            'OK',
            'Gitlab backup for yesterday is found'
        )
