# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.sandboxsdk import environments
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.projects.deploy.CreateDeployDockerRelease import CreateDeployDockerRelease
from sandbox.projects.adfox.adfox_ui.util.duty_tools import DutyTools
from sandbox.projects.adfox.adfox_ui.util.config import GitConfig
from sandbox.projects.adfox.adfox_ui.deploy.adfox_ui_deploy_task import AdfoxUiDeployTask, \
    Stage, Secrets
from sandbox.projects.adfox.adfox_ui.metrics import AnalyzableTask

import time

WAIT_DEPLOY_TIMEOUT = 30 * 60
DEPLOY_TIMEOUT = 8 * 60


class AdfoxUiBuild(AnalyzableTask):

    class Parameters(AnalyzableTask.Parameters):
        git_branch = sdk2.parameters.String('Git branch', default=GitConfig.RELEASE_CANDIDATE_PREFIX, required=True)

        with sdk2.parameters.RadioGroup('Stage', required=True) as stage:
            stage.values[Stage.PRE_PROD] = stage.Value(Stage.PRE_PROD, default=True)
            stage.values[Stage.AUTO_TEST] = stage.Value(Stage.AUTO_TEST, default=False)

        attempts = sdk2.parameters.Integer('Attempts to wait building', default='180', required=True)

    def on_execute(self):
        super(AdfoxUiBuild, self).on_execute()
        branch = self.Parameters.git_branch

        if branch == GitConfig.RELEASE_CANDIDATE_PREFIX:
            branch = DutyTools().get_release_candidate_version()

        for i in range(self.Parameters.attempts):
            logging.info("Waiting for pipeline. Iteration %d." % i)
            status = DutyTools().get_build_status(branch, self.Parameters.stage)
            if status == 'success':
                logging.info("Done!")
                return
            time.sleep(60)

        raise RuntimeError('Build error')

    def get_release_id(self):
        return self.Parameters.git_branch


class AdfoxUiDeploy(binary_task.LastBinaryTaskRelease, AnalyzableTask):
    """ Деплоит окружение """

    class Requirements(sdk2.Task.Requirements):
        # В продакшене использовать контейнер отсюда
        # https://sandbox.yandex-team.ru/task/836669544
        container_resource = 773239891

        # В девеле копируем контейнер локально и записываем ниже
        # ./sandbox add_resource --id 773239891
        # Подробнее https://wiki.yandex-team.ru/sandbox/quickstart/#import-from-production
        # container_resource = 71
        dns = ctm.DnsType.DNS64
        environments = (
            environments.PipEnvironment('pymysql', use_wheel=True, version='0.9.3'),
        )
        privileged = True

    class Parameters(AnalyzableTask.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)
        git_branch = sdk2.parameters.String('Git branch', default=GitConfig.RELEASE_CANDIDATE_PREFIX, required=True)
        clients = sdk2.parameters.String('Clients list', default='1,273063,233605,239800',
                                         required=True)  # root, ExternalMonetization
        stage_id = sdk2.parameters.String('Stage ID', default='0', required=True)

        with sdk2.parameters.Group("YAV secrets") as yav_secrets:
            oauth_secret = sdk2.parameters.YavSecret(
                'Secret for OAuth',
                default='sec-01djcsd32jjt8cz9ppa3ntpvwn',
                required=True,
            )
            root_secret = sdk2.parameters.YavSecret(
                'Secret for root password',
                default='sec-01e7mzpbd03fgkxzxsar79p84h',
                required=True,
            )

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "ADFOX_UI_DEPLOY",
                "released": self.Parameters.ext_params.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    def on_enqueue(self):
        super(AdfoxUiDeploy, self).on_enqueue()

    def get_deploy_task(self):
        # type: () -> AdfoxUiDeployTask

        secrets = sdk2.yav.Yav(
            oauth=self.Parameters.oauth_secret,
            root=self.Parameters.root_secret,
        )
        secrets_obj = Secrets(
            oauth_qloud_token=secrets.oauth['oauth-qloud'],
            docker_registry_token=secrets.oauth['docker-registry'],
        )
        return AdfoxUiDeployTask(
            secrets_obj,
            self.Parameters.git_branch,
            self.Parameters.stage_id,
            self
        )

    def on_execute(self):
        super(AdfoxUiDeploy, self).on_execute()
        if not self.Context.release_task:
            logging.info("Init task.")
            deploy_task = self.get_deploy_task()
            deploy_task.execute()
        task = self.find(id=self.Context.release_task, status=ctt.Status.SUCCESS).limit(1).first()
        if task is None:
            raise Exception("Sub task failed. See task id #%s." % self.Context.release_task)

        logging.info("Done!")

    def start_docker_release(self, images, release_type='stable', wait_stages=None):
        release_task = CreateDeployDockerRelease(
            self,
            description="Child of task {}".format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            __requiremetns__={"tasks_resource": self.Requirements.tasks_resource},
        )
        release_task.Parameters.yp_token_yav_secret = self.Parameters.oauth_secret
        release_task.Parameters.yp_token_yav_secret_key = 'oauth-dctl-yp'
        release_task.Parameters.release_type = release_type
        release_task.Parameters.docker_images = images
        if wait_stages:
            release_task.Parameters.wait_deploy_tickets = True
            release_task.Parameters.wait_stages = wait_stages
        self.Context.release_task = release_task.save().enqueue().id
        logging.info('wait deploy...')
        raise sdk2.WaitTask(
            [self.Context.release_task],
            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
            timeout=WAIT_DEPLOY_TIMEOUT,
            wait_all=True
        )
