# -*- coding: utf-8 -*-
import logging
from sandbox.projects.adfox.adfox_ui.util.release_integration import AdfoxReleaseIntegration
from sandbox.projects.adfox.adfox_ui.util.process import Process
from sandbox.projects.adfox.adfox_ui.util.resource_helper import ResourceHelper
from datetime import datetime

RELEASE_CANDIDATE_REPO = 'adfox/interfaces/pre_production/login'
RELEASE_CANDIDATE_RELEASE_TYPE = 'rc'
RELEASE_CANDIDATE_STAGE = 'adfox-crowdtest'

RESOURCE_PREFIX = 'sandbox/projects/adfox/adfox_ui/deploy/'
INTERFACES_TAG_SH = 'interfaces_tag.sh'


def image_name(repo, tag):
    return '{}:{}'.format(repo, tag)


class Stage(object):
    def __init__(self):
        pass

    AUTO_TEST = 'autotest'
    PRE_PROD = 'preprod'


class Secrets:
    def __init__(self, oauth_qloud_token, docker_registry_token):
        self.oauth_qloud_token = oauth_qloud_token
        self.docker_registry_token = docker_registry_token


class AdfoxUiDeployTask:

    def __init__(self, secrets, git_branch, stage_id, task=None):
        # type: (Secrets, List[int], str, str, AdfoxUiDeploy) -> None

        self.secrets = secrets
        self.git_branch = git_branch
        self.stage_id = stage_id
        self.task = task
        self.release_tag = None
        self.ts_tag = None

    def get_release_tag(self):
        if not self.release_tag:
            self.release_tag = AdfoxReleaseIntegration.resolve_branch(self.git_branch)
        return self.release_tag

    def get_timestamped_tag(self):
        if not self.ts_tag:
            self.ts_tag = '{}-{}'.format(self.get_release_tag(), datetime.now().strftime('%y%m%d%H%M%S'))
        return self.ts_tag

    def start_yadeploy(self, repo_list, release_type='stable', wait_stages=None):
        if not self.task:
            logging.info('Release in YaDeploy not available without sandbox context. '
                         'Update images to tag {} manually in YaDeploy')
            return

        def get_tag_by_repo(repo):
            return self.get_timestamped_tag()

        images = map(lambda repo: image_name(repo, get_tag_by_repo(repo)), repo_list)
        self.task.start_docker_release(images, release_type, wait_stages)

    def create_interfaces_tag(self, interfaces_repo):
        with ResourceHelper(RESOURCE_PREFIX) as rh:
            sh_file_path = rh.extract(INTERFACES_TAG_SH)
            command = '/bin/bash {} {} {} {} {}'.format(
                sh_file_path,
                self.get_release_tag(),
                self.get_timestamped_tag(),
                self.secrets.docker_registry_token,
                interfaces_repo
            )
            process = Process(task=self.task, name='interfaces_tag')
            process.run(command)

    def deploy_pre_prod(self):
        self.create_interfaces_tag(RELEASE_CANDIDATE_REPO)
        self.start_yadeploy(
            repo_list=[RELEASE_CANDIDATE_REPO],
            release_type='testing',
            wait_stages=[RELEASE_CANDIDATE_STAGE]
        )

    def execute(self):
        logging.info("Deploying pre-prod.")
        self.deploy_pre_prod()

    def normalize(self, name):
        if self.stage_id in ['0', '']:
            return name
        return '{}-{}'.format(name, self.stage_id)

    def get_stage_name(self, base_name):
        return self.normalize(base_name)
