# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2
from sandbox.projects.common.juggler import jclient
from ..util.config import DbConfig, JugglerConfig
from sandbox.sandboxsdk import environments
import datetime

DEFAULT_JUGGLER_SERVICE = 'adfox-clickhouse-object-active-days'

MAX_DAYS_BEFORE = 14

tables_config = [
    {
        'counters_table': 'banner_hourly_counters',
        'counters_daily_table': 'banner_daily_counters',
        'active_day_table': 'banner_active_day',
        'active_day_id_field': 'banner_id'
    },
    {
        'counters_table': 'campaign_hourly_counters',
        'counters_daily_table': 'campaign_daily_counters',
        'active_day_table': 'campaign_active_day',
        'active_day_id_field': 'campaign_id'
    },
    {
        'counters_table': 'supercampaign_hourly_counters',
        'counters_daily_table': 'supercampaign_daily_counters',
        'active_day_table': 'supercampaign_active_day',
        'active_day_id_field': 'supercampaign_id'
    },
]


class AdfoxUiObjectActiveDay(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        day = sdk2.parameters.String('Day (Максимум 14 дней назад)', default='', required=False)
        with sdk2.parameters.RadioGroup("Date") as processing_date:
            processing_date.values.TODAY = processing_date.Value(default=True)
            processing_date.values.YESTERDAY = None

        mysql_host = sdk2.parameters.String('MYSQL host', default=DbConfig.DEFAULT_MYSQL_HOST, required=True)
        mysql_user = sdk2.parameters.String('MYSQL user', default=DbConfig.DEFAULT_MYSQL_USER, required=True)
        mysql_pass_secret = sdk2.parameters.String('MYSQL password secret',
                                                   default=DbConfig.DEFAULT_MYSQL_PASSWORD_SECRET,
                                                   required=True)

        with sdk2.parameters.Group("Juggler Parameters") as juggler_block:
            juggler_host = sdk2.parameters.String('Juggler host', default=JugglerConfig.DEFAULT_JUGGLER_HOST, required=True)
            juggler_service = sdk2.parameters.String(
                'Juggler service',
                default=DEFAULT_JUGGLER_SERVICE,
                required=True
            )

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('pymysql', use_wheel=True, version='0.9.3'),
        )

    db = None
    tables_config = None
    day = None
    remove_day = None

    def init_task(self):
        from sandbox.projects.adfox.adfox_ui.util.db_conn import DbConnection

        mysql_pass_secret = sdk2.yav.Secret(self.Parameters.mysql_pass_secret)

        self.db = DbConnection({
            'host': self.Parameters.mysql_host,
            'user': self.Parameters.mysql_user,
            'passwd': mysql_pass_secret.data()[self.Parameters.mysql_user],
        })

        day_today = datetime.date.today()
        if self.Parameters.day:
            day_param = datetime.datetime.strptime(self.Parameters.day, '%Y-%m-%d').date()
            diff = day_today - day_param

            if diff.days < 0 or diff.days > MAX_DAYS_BEFORE:
                raise Exception('Введите дату менее 14 дней назад')
            else:
                self.day = day_param.strftime('%Y-%m-%d')
        elif self.Parameters.processing_date == 'TODAY':
            self.day = datetime.date.today().strftime('%Y-%m-%d')
        elif self.Parameters.processing_date == 'YESTERDAY':
            self.day = (datetime.date.today() - datetime.timedelta(days=1)).strftime('%Y-%m-%d')
        else:
            raise Exception('Некорректная дата')

        self.remove_day = (datetime.date.today() - datetime.timedelta(days=3 * 365)).strftime('%Y-%m-%d')

    def on_execute(self):

        self.init_task()

        logging.info('PROCESSING DAY:' + self.day)
        self.set_info('PROCESSING DAY:' + self.day)

        logging.info('REMOVE DAY:' + self.remove_day)
        self.set_info('REMOVE DAY:' + self.remove_day)

        for table_config in tables_config:
            self.remove_old_days(table_config['active_day_table'])

            active_ids = self.select_active_object_ids(table_config['counters_table'],
                                                       table_config['counters_daily_table'])

            # 12345, 23456 -> (12345, '2020-01-01'), (23456, '2020-01-01')
            id_day_str = self.make_ids_day_insert_string(active_ids)

            self.insert_ignore_active_ids(table_config['active_day_table'],
                                          table_config['active_day_id_field'],
                                          id_day_str)

        self.juggler_notify_done()

        logging.info('FINISH')
        self.set_info('FINISH')

    def remove_old_days(self, active_day_table):
        logging.info('START REMOVING OLD DAYS')
        self.set_info('START REMOVING OLD DAYS')

        remove_old_query = """
                        DELETE FROM adfox.{active_day_table}
                        WHERE day <= '{remove_day}';
                    """.format(remove_day=self.remove_day, active_day_table=active_day_table)
        self.db.execute_queries([remove_old_query])

        logging.info('FINISHED REMOVING OLD DAYS')
        self.set_info('FINISHED REMOVING OLD DAYS')

    def select_active_object_ids(self, counters_table, counters_table_daily):
        select_active_ids_query = """
                                SELECT distinct(id) FROM adfox.{counters_table} ct
                                WHERE ct.time >= '{day} 00:00:00' AND ct.time <= '{day} 23:00:00'
                                AND (ct.impressions > 0 OR ct.clicks > 0);
                            """.format(day=self.day, counters_table=counters_table)

        logging.info('SELECTING IDS TO INSERT FROM HOURLY COUNTERS')
        self.set_info('SELECTING IDS TO INSERT FROM HOURLY COUNTERS')

        result_hourly = self.db.fetch_all(select_active_ids_query)

        logging.info('selected #:' + str(len(result_hourly)))

        select_active_ids_query_daily = """
                                       SELECT distinct(id) FROM adfox.{counters_table_daily} ct
                                       WHERE ct.day = '{day}'
                                       AND (ct.impressions > 0 OR ct.clicks > 0);
                                   """.format(day=self.day,
                                              counters_table_daily=counters_table_daily)

        logging.info('SELECTING IDS TO INSERT FROM DAILY COUNTERS')
        self.set_info('SELECTING IDS TO INSERT FROM DAILY COUNTERS')

        result_daily = self.db.fetch_all(select_active_ids_query_daily)

        logging.info('selected #:' + str(len(result_daily)))

        result = set(result_hourly + result_daily)

        logging.info('IDS NUMBER TO INSERT: ' + str(len(result)))

        if len(result) == 0:
            raise Exception('Нет данных счетчиков за переданную дату')

        return result

    def insert_ignore_active_ids(self, active_day_table, active_day_id_field, values_str):
        insert_query = """
                                INSERT IGNORE INTO adfox.{active_day_table}
                                ({active_day_id_field}, day) VALUES
                                {values_str};
                            """.format(values_str=values_str,
                                       active_day_table=active_day_table,
                                       active_day_id_field=active_day_id_field)

        logging.info('INSERTING IDS')
        self.db.execute_queries([insert_query])

    def make_ids_day_insert_string(self, active_ids):
        return ', '.join(map(lambda obj: "({obj_id}, '{day}')".format(obj_id=obj[0], day=self.day), active_ids))

    def juggler_notify_done(self):
        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            'OK',
            'Active days successfully updated'
        )
