# -*- coding: utf-8 -*-

import requests
import time
import logging

logger = logging.getLogger('curl')
logger.setLevel(logging.DEBUG)


def retry(func):
    def wrapper(self, query, **kwargs):
        for _ in range(self.attempts):
            try:
                result = func(self, query, **kwargs)
                if result.status_code != 200:
                    logger.info('Retry (Incorrect status code)')
                    time.sleep(self.wait_time)
                    self.wait_time *= self.wait_time_multiplier
                    continue
                return result
            except requests.exceptions.ConnectionError:
                logger.info('Retry (Connection error)')
        raise RuntimeError

    return wrapper


class CurlClient:

    def __init__(self, host, attempts=3, wait_time=10, wait_time_multiplier=2, default_headers=None):
        if default_headers is None:
            default_headers = {}
        self.attempts = attempts
        self.wait_time = wait_time
        self.wait_time_multiplier = wait_time_multiplier
        self.host = host
        self.default_headers = default_headers

    @retry
    def get(self, query, headers=None, **kwargs):
        if headers is None:
            headers = {}
        headers.update(self.default_headers)
        return requests.get('%s%s' % (self.host, query), headers=headers, **kwargs)

    def post(self, query, json, headers=None):
        if headers is None:
            headers = {}
        headers.update(self.default_headers)
        return requests.post('%s%s' % (self.host, query), json=json, headers=headers)

    def patch(self, query, json, headers=None):
        if headers is None:
            headers = {}
        headers.update(self.default_headers)
        return requests.patch('%s%s' % (self.host, query), json=json, headers=headers)

    def delete(self, query, headers=None):
        if headers is None:
            headers = {}
        headers.update(self.default_headers)
        return requests.delete('%s%s' % (self.host, query), headers=headers)
