# -*- coding: utf-8 -*-
import hashlib
from distutils.version import LooseVersion
import requests
import logging

MANIFEST_V2_CONTENT_TYPE = 'application/vnd.docker.distribution.manifest.v2+json'


class Registry(object):
    def __init__(self, repo, oauth_token):
        self.repo = repo
        self.headers = {
            'Authorization': 'OAuth %s' % oauth_token,
            'Accept': MANIFEST_V2_CONTENT_TYPE,
        }

    def request(self, method, path, headers=None, data=None):
        headers = self.headers if not headers else dict(self.headers, **headers)
        return requests.request(
            method,
            'https://registry.yandex.net/v2/%s%s' % (self.repo, path),
            headers=headers,
            data=data
        )

    def version(self, v):
        return Version(self, v)

    @property
    def versions(self):
        resp = self.request('GET', '/tags/list')
        versions = resp.json().get('tags', [])
        versions = sorted(versions, key=lambda x: LooseVersion(x))
        return [Version(self, version) for version in versions]


class Version(object):
    def __init__(self, registry, version):
        self.version = version
        self.registry = registry

    def __repr__(self):
        return 'Version(%s @ %s)' % (self.registry.repo, self.version)

    def get_manifest(self):
        resp = self.registry.request('GET', '/manifests/%s' % self.version)
        return resp.content

    @property
    def digest(self):
        m = hashlib.sha256()
        m.update(self.get_manifest())
        return m.hexdigest()


class DockerRegistry(object):
    def __init__(self, repo, token):
        self.repo = repo
        self.token = token
        self.registry = Registry(self.repo, self.token)

    def get_hash(self, tag):
        versions = self.registry.versions
        version_hash = None
        for version in versions:
            if version.version == tag:
                version_hash = version.digest
                break
        if version_hash is None:
            raise Exception('Invalid tag')
        return version_hash

    def get_repo(self):
        return self.repo

    def copy_tag(self, from_tag, to_tag):
        logging.info('copy_tag from {} to {}'.format(from_tag, to_tag))
        manifest = self.registry.version(from_tag).get_manifest()
        return self.registry.request(
            'PUT',
            '/manifests/%s' % to_tag,
            headers={
                'Content-type': MANIFEST_V2_CONTENT_TYPE,
            },
            data=manifest
        )
