# -*- coding: utf-8 -*-

from .curl import CurlClient, retry
import json
import logging


class DutyTools:

    # API_URL = 'https://ui-tools-devel-2.devel.qloud.adfox.net'
    # API_URL = 'https://ui-tools-devel.devel.qloud.adfox.net'
    API_URL = 'https://ui-tools.adfox.yandex-team.ru'

    def __init__(self):
        self.curl = CurlClient(self.API_URL)

    def get_release_version(self):
        return self.curl.get('/git/release-version').content

    def get_release_candidate_version(self):
        return self.curl.get('/git/release-candidate-version').content

    def get_build_status(self, branch, stage):
        return self.curl.get('/git/build-status?branch={}&stage={}'.format(branch, stage)).content

    def get_autotest_info(self, identifier):
        return json.loads(self.curl.get('/autotest/%s/info' % identifier).content)

    def autotest_notify_report(self, task):
        self.curl.get('/autotest/{}/notify?{}'.format(task.identifier, self.__get_autotest_notify_params(task)))

    def autotest_locked(self):
        return json.loads(self.curl.get('/autotest/locked').content)

    def autotest_notify_start(self, task):
        try:
            self.curl.get('/autotest/notify_start?{}'.format(self.__get_autotest_notify_params(task)))
        except Exception as e:
            logging.error(e)

    def autotest_notify_retry(self, task):
        try:
            self.curl.get('/autotest/{}/notify_retry?{}'.format(task.identifier, self.__get_autotest_notify_params(task)))
        except Exception as e:
            logging.error(e)

    def autotest_notify_success(self, task):
        try:
            self.curl.get('/autotest/{}/notify_success?{}'.format(task.identifier, self.__get_autotest_notify_params(task)))
        except Exception as e:
            logging.error(e)

    def autotest_notify_fail(self, task):
        try:
            self.curl.get('/autotest/{}/notify_fail?{}'.format(task.identifier, self.__get_autotest_notify_params(task)))
        except Exception as e:
            logging.error(e)

    def autotest_check_report(self, identifier):
        self.curl.get('/autotest/%s/' % identifier).status_code

    def __get_autotest_notify_params(self, task):
        author = ''
        # Если запускается из нашей группы, то отсылаем автору, если нет, то определяем из имени ветки или если pre-release-login, то дежурный
        if task.owner in ['ADFOX', 'ADFOX-UI']:
            author = task.author

        return 'branch={}&task_id={}&author={}'.format(task.Parameters.gitBranch, task.id, author)

    def send_common_metrics(self, metrics):
        self.send_metrics('common_metrics', metrics)

    def send_assessors_metrics(self, metrics):
        self.send_metrics('assessors_metrics', metrics)

    def send_assessors_cases(self, metrics):
        self.send_metrics('assessors_cases', metrics)

    def send_autotest_metrics(self, metrics):
        self.send_metrics('autotest_metrics', metrics)

    def send_coverage_maps(self, metrics):
        self.send_metrics('coverage_maps', metrics)

    # Отправка метрик в duty-tools
    def send_metrics(self, type, metrics):
        try:
            logging.info('duty_tools send: {}'.format(type))
            logging.info('metrics: {}'.format(metrics[:10]))
            self.curl.post('/metrics', {
                'data_type': type,
                'data': metrics
            })
        except Exception as e:
            logging.error(e.message)

    def start_autotests(self, branch=None, front_tag=None, backend_tag=None, java_tag=None):
        """
        @param branch: string
        @return: int[]
        """
        response = self.curl.get(
            '/release/start-autotests',
            params={
                'branch': branch,
                'frontend_tag': front_tag,
                'backend_tag': backend_tag,
                'java_tag': java_tag,
            }
        ).json()
        logging.info('response json:')
        logging.info(json.dumps(response))
        jobs = response.get('jobs')
        job_ids = [job.get("id") for job in jobs]
        logging.info('job ids:')
        logging.info(json.dumps(job_ids))
        return job_ids

    def get_gitlab_jobs(self, ids):
        logging.info('get_gitlab_jobs ids:')
        logging.info(json.dumps(ids))
        response = self.curl.get('/git/get_jobs', params={'ids': ",".join(map(str, ids))}).json()
        logging.info('get_gitlab_jobs response json:')
        logging.info(json.dumps(response))
        return response.get('jobs')

    def get_autotest_report_url(self, job_id):
        response = self.curl.get('/git/job/{}/report'.format(job_id))
        return response.content if response.ok else None

    def mark_release(self, branch, event):
        try:
            retry(self.curl.post('/release/mark-release?branch={}&event={}'.format(branch, event)))
        except Exception as e:
            logging.error(e)

dutyTools = DutyTools()
