# -*- coding: utf-8 -*-

import requests
import time
from .curl import CurlClient


class Component:
    def __init__(self, name='', repo='', hash=''):
        self.name = name
        self.repo = repo
        self.hash = hash


class Qloud:
    """
    Деплой и получение информации  об окружениях в qloud-е.
    """

    API_URL = 'https://qloud-ext.yandex-team.ru/api/v1/environment'
    LOAD_API_URL = API_URL + '/fast-deploy/{environment}/{version}'
    INFO_API_URL = '/current/{environment}'
    DUMP_URL = '/dump/{environment}'
    FULL_DEPLOY_URL = API_URL + '/upload/return-header?targetState=DEPLOYED&skipIfNoChange=false'

    STATUS_DEPLOYED = 'DEPLOYED'

    def __init__(self, token, environment, wait_time=10):
        """
        :param token: string
        :param environment: string
        :param wait_time: int
        """
        self.token = token
        self.environment = environment
        self.wait_time = wait_time
        self.curl = CurlClient(self.API_URL)

    def get_info(self):
        result = self.curl.get(Qloud.INFO_API_URL.format(environment=self.environment), headers=self.get_headers())
        return result.json()

    def get_dump(self):
        result = self.curl.get(Qloud.DUMP_URL.format(environment=self.environment), headers=self.get_headers())
        return result.json()

    def full_deploy(self, json):
        result = requests.post(Qloud.FULL_DEPLOY_URL, headers=self.get_headers(), json=json)
        if result.status_code != 204:
            return result.status_code
        else:
            return None

    def fast_deploy(self, components):
        """
        Быстрый деплой компонентов, обновляет только докер образ.
        :param components: Component[]
        :return: status_code
        """

        version = self.get_info()['version']

        url = Qloud.LOAD_API_URL.format(environment=self.environment, version=version)

        data = []
        for component in components:
            data.append({
                'componentName': component.name,
                'properties': {
                    'repository': component.repo,
                    'hash': component.hash
                }
            })

        result = requests.post(url, headers=self.get_headers(), json=data)

        if result.status_code != 204:
            return result.status_code
        else:
            return None

    def force_update(self, component_name):
        """
        Обновляет / передеплаивает компонент, даже если в нем не было изменений с прошлого деплоя.
        :param component_name: Название компонента, который нужно насильно обновить.
        :return:
        """
        dump = self.get_dump()

        # Ищем компонент в json-e
        dump_to_update_index = -1
        i = 0
        for component_dump in dump['components']:
            if component_dump['componentName'] == component_name:
                dump_to_update_index = i
                break
            i += 1

        # Делаем подмену переменной в env, чтобы qloud считал, что окружение изменилось и передеполил его.
        if 'env_for_update_1' in dump['components'][dump_to_update_index]['environmentVariables']:
            dump['components'][dump_to_update_index]['environmentVariables']['env_for_update_2'] = 1
            dump['components'][dump_to_update_index]['environmentVariables'].pop('env_for_update_1', None)
        else:
            dump['components'][dump_to_update_index]['environmentVariables']['env_for_update_1'] = 1
            dump['components'][dump_to_update_index]['environmentVariables'].pop('env_for_update_2', None)
        print(dump['components'][dump_to_update_index]['environmentVariables'])

        # Запускаем деплой с обновленным json-ом.
        return self.full_deploy(dump)

    def wait_deploying(self, callback_while_waiting):
        while self.get_info()['status'] != Qloud.STATUS_DEPLOYED:
            callback_while_waiting()
            time.sleep(self.wait_time)

    def get_headers(self):
        return {'authorization': 'OAuth ' + self.token, 'Content-Type': 'application/json'}
