# -*- coding: utf-8 -*-

from sandbox.sdk2 import Task, parameters
from sandbox.projects.adfox.adfox_ui.util.process import Process
from sandbox.projects.adfox.adfox_ui.util.config import GitConfig
from sandbox.sandboxsdk import errors
import sandbox.common.types.misc as ctm
import os
import time
import hashlib
import yaml
from pandas.core.common import flatten


class AdfoxUiYamlValidate(Task):
    """ Валидирует yaml-ы в Adfox и ПИ """

    class Requirements(Task.Requirements):
        # В продакшене использовать контейнер отсюда
        # https://sandbox.yandex-team.ru/task/692154236/view
        container_resource = 1541761107

        # В девеле копируем контейнер локально и записываем ниже
        # ./sandbox add_resource --id 1541761107
        # Подробнее https://wiki.yandex-team.ru/sandbox/quickstart/#import-from-production
        # container_resource = 34
        dns = ctm.DnsType.DNS64

    class Parameters(Task.Parameters):
        identifier = parameters.String(
            'Identifier',
            default=None,
            required=False,
            description="Идентификатор (заполнять не нужно, сгенерируется автоматически)"
        )

        gitBranchAdfox = parameters.String(
            'Git branch Adfox',
            default=None,
            required=True,
            description="Ветка git Adfox"
        )

        gitBranchYharnam = parameters.String(
            'Git branch Yharnam',
            default=None,
            required=True,
            description="Ветка git yharnam"
        )

    def on_enqueue(self):
        if not self.parent:
            if self.Parameters.identifier:
                raise Exception("You can't use identifier field, use clear value")

    def on_prepare(self):
        process = Process(task=self, name='git')

        process.run("git clone http://git.adfox.ru/adfox/interfaces.git")
        process.run("cd interfaces && git checkout {} && git pull".format(self.Parameters.gitBranchAdfox))

        process.run("git clone git@github.yandex-team.ru:partner/yharnam.git")
        process.run("cd yharnam && git checkout {} && git pull".format(self.Parameters.gitBranchYharnam))

        if self.Parameters.gitBranch != 'pre-release-login' and not self.Parameters.gitBranch.startswith(GitConfig.RELEASE_CANDIDATE_PREFIX):
            process.run("cd interfaces && git config user.name 'robot-adfox-ui'")
            process.run("cd interfaces && git config user.email 'robot-adfox-ui@yandex-team.ru'")
            process.run("cd interfaces && git merge origin/pre-release-login --no-ff")

    def on_execute(self):
        adfox_branch = str(self.Parameters.gitBranchAdfox)
        yharnam_branch = str(self.Parameters.gitBranchYharnam)

        self.identifier = self.Parameters.identifier or hashlib.md5(
            (adfox_branch + ':' + yharnam_branch + ':' + str(int(time.time()))).encode()
        ).hexdigest()

        self.Context.identifier = self.identifier
        self.Context.save()

        dirs = [
            'interfaces/.tests/tests/hermione',
            'yharnam/test/autotests/suites'
        ]

        yamls = []

        for dir in dirs:
            for root, _, files in os.walk(dir):
                for file in files:
                    if file.endswith('.yml'):
                        yamls.append(os.path.join(root, file))

        keys = set()
        parents = set()

        for y in yamls:
            test = yaml.safe_load(open(y, 'r'))
            if 'specs' not in test:
                continue
            for spec_key, spec in test['specs'].items():
                for item in spec:
                    if 'params' not in item:
                        continue
                    params = item['params']
                    if 'key' in params:
                        for k in flatten([params['key']]):
                            keys.add(k)
                    if 'parent' in params:
                        for par in flatten([params['parent']]):
                            parents.add(par)

        problems = parents.difference(keys)
        print("Problems found: in keys/parents: %s" % problems)
        if len(problems):
            raise errors.SandboxTaskFailureError('Test failure')
