# -*- coding: utf-8 -*-
import os
from os.path import join as join_path, realpath, dirname
from sandbox import sdk2
import logging
import sys
import time
import datetime
from sandbox.sandboxsdk.process import run_process
from shutil import copyfile


LXC_CONTAINER_ID = 883907995
TMP_DIR = '/tmp'
MY_DIR = '/etc/mysql'
MY_CNF = 'my.cnf'
MYSQL_SLAVE_CONFIG_DIR = '.mysql'
MYSQL_SLAVE_CONFIG_FILE = 'config'


class AdfoxBackupValidate(sdk2.Task):

    class Requirements(sdk2.Requirements):
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            'LXC Container with mysql-5.6',
            default_value=LXC_CONTAINER_ID,
            platform='linux_ubuntu_14.04_trusty',
            required=True,
        )
        url = sdk2.parameters.String('s3 mds url', default='http://s3.mds.yandex.net', required=True)
        bucket_name = sdk2.parameters.String('s3 bucket name', default='adfox', required=True)
        backup_host = sdk2.parameters.String('backup host name', default='bkp1.myt.adfox.ru', required=True)
        check_prev_day = sdk2.parameters.Bool("Check previous day backup")
        with check_prev_day.value[False]:
            backup_date = sdk2.parameters.String("backup date in '%Y%m%d' format", default='20190101', required=True)
        innodb_memory = sdk2.parameters.String('memory limit for innobackupex', default='8G', required=False)
        tmp_dir = sdk2.parameters.String('tmp dir', default=TMP_DIR, required=False)
        slave_wait = sdk2.parameters.String('checking slave status for...in minutes', default=30, required=False)
        with sdk2.parameters.Group("settings for mysql master connection") as db_params:
            mysql_host = sdk2.parameters.String('Database host', default='db.adfox.ru', required=True)
            mysql_user = sdk2.parameters.String('Database user', default='repl', required=True)
            db_password_vault = sdk2.parameters.String('Vault for password',
                                                       default='ADFOX_MYSQL_REPLICATION_PASS')
        with sdk2.parameters.Group("aws credentials") as aws_credentials:
            s3_id_vault = sdk2.parameters.String('Vault for s3 id',
                                                 default='ADFOX_S3_ACCESS_KEY_ID')
            s3_secret_vault = sdk2.parameters.String('Vault for s3 secret',
                                                     default='ADFOX_S3_ACCESS_KEY_SECRET')

    def on_execute(self):
        logging.getLogger().setLevel(logging.DEBUG)
        handler = logging.StreamHandler(sys.stdout)
        handler.setLevel(logging.DEBUG)
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        handler.setFormatter(formatter)
        logging.getLogger().addHandler(handler)
        logging.info('Starting script...')
        xtrabackup_script_path = join_path(dirname(realpath(__file__)), 'xtrabackup_restore.py')
        backup_validation_script_path = join_path(dirname(realpath(__file__)), 'backup_validation.py')
        my_cfn_path = join_path(dirname(realpath(__file__)), MY_CNF)
        my_cfn_dest = join_path(MY_DIR, MY_CNF)
        copyfile(my_cfn_path, my_cfn_dest)
        xtrabackup_env = dict(os.environ)
        xtrabackup_env["AWS_ACCESS_KEY_ID"] = sdk2.Vault.data('ADFOX', self.Parameters.s3_id_vault)
        xtrabackup_env["AWS_SECRET_ACCESS_KEY"] = sdk2.Vault.data('ADFOX', self.Parameters.s3_secret_vault)
        mysql_env = dict(os.environ)
        mysql_env['MYSQL_HOST'] = self.Parameters.mysql_host
        mysql_env['MYSQL_USER'] = self.Parameters.mysql_user
        mysql_env['MYSQL_PASSWORD'] = sdk2.Vault.data('ADFOX', self.Parameters.db_password_vault)
        tmp_dir = self.Parameters.tmp_dir
        if not os.path.isdir(tmp_dir):
            os.makedirs(tmp_dir)
        if self.Parameters.check_prev_day:
            prev_date = datetime.datetime.today()
            backup_date = prev_date.strftime('%Y%m%d')
        else:
            backup_date = self.Parameters.backup_date
        logging.info('launching xtrabackup_restore')
        run_process([
            xtrabackup_script_path,
            "--url='{}'".format(self.Parameters.url),
            "--bucket-name='{}'".format(self.Parameters.bucket_name),
            "--backup-host='{}'".format(self.Parameters.backup_host),
            "--backup-date='{}'".format(backup_date),
            "--tmp_dir='{}'".format(tmp_dir),
            "--cnf-file='{}'".format(my_cfn_dest),
            "--innodb-memory='{}'".format(self.Parameters.innodb_memory),
        ], log_prefix='xtrabackup_restore', environment=xtrabackup_env, shell=True)
        logging.info('xtrabackup_restore is done, sleeping for 30 seconds to make sure mysql started')
        time.sleep(30)
        logging.info('launching backup_validation')
        run_process([
            backup_validation_script_path,
            "--slave-wait='{}'".format(self.Parameters.slave_wait),
            "--cnf-file='{}'".format(my_cfn_dest),
            "--backup-date='{}'".format(backup_date),
        ], log_prefix='backup_validation', environment=mysql_env, shell=True)
        logging.info('all done')
