#!/usr/bin/env python
import os
import logging
import ConfigParser
import pymysql
import sys
import json
import time
from argparse import ArgumentParser

SLAVE_SLEEP_TIME = 60


class ConfigError(Exception):
    pass


class LastUpdateCheckError(Exception):
    pass


class BackupValidation:
    def __init__(self, **kwargs):
        self.params = kwargs
        self.db_conn = None

    def get_slave_pos(self):
        config = ConfigParser.ConfigParser(allow_no_value=True)
        config.read(self.params['cnf_file'])
        datadir = config.get('mysqld', 'datadir')

        # in xtrabackup_slave_info file info about position of slave
        slave_info_file = os.path.join(datadir, 'xtrabackup_slave_info')
        with open(slave_info_file, "r") as f:
            slave_pos = f.read()
        logging.info(slave_pos)

        # in first row info about gtid positions
        slave_pos_gtid = slave_pos.split(';')[0]
        logging.info(slave_pos_gtid)

        return slave_pos_gtid

    def get_db_conn(self):
        if self.db_conn is None or not self.db_conn.open:
            self.db_conn = pymysql.connect(**self.get_local_db_config())
        return self.db_conn

    # config hardcoded for now: backup mysql always only local and required root priv
    @staticmethod
    def get_local_db_config():
        return {
            'host': 'localhost',
            'user': 'root',
            'passwd': '',
        }

    def create_slave(self):
        config = self.get_master_config()
        query_to_set_gtid = self.get_slave_pos()
        query = """CHANGE MASTER TO MASTER_AUTO_POSITION=1, MASTER_HOST='{host}', MASTER_USER='{user}', MASTER_PASSWORD='{password}',
             MASTER_CONNECT_RETRY = 5"""\
            .format(host=config['host'], user=config['user'],
                    password=config['password'],)
        cur = self.get_db_conn().cursor()
        cur.execute(query_to_set_gtid)
        cur.execute(query)
        query = "start slave"
        cur.execute(query)
        cur.close()
        time.sleep(5)

    def get_slave_status(self):
        cur = self.get_db_conn().cursor(pymysql.cursors.DictCursor)
        query = 'show slave status'
        cur.execute(query)
        res = cur.fetchall()
        return res[0]

    def check_slave_errors(self):
        slave_status = self.get_slave_status()
        for field in self.get_bad_slave_error_fields():
            if slave_status[field]:
                logging.error(json.dumps(slave_status))
                return False
        if slave_status['Slave_IO_Running'] != 'Yes' or slave_status['Slave_SQL_Running'] != 'Yes':
            logging.error(json.dumps(slave_status))
            return False
        logging.info('slave running file {} at pos {}'.format(slave_status['Relay_Master_Log_File'],
                                                              slave_status['Exec_Master_Log_Pos']))
        return True

    @staticmethod
    def get_bad_slave_error_fields():
        return [
            'Last_IO_Errno',
            'Last_IO_Error',
            'Last_SQL_Errno',
            'Last_SQL_Error',
        ]

    def get_last_update_queries(self):
        date = self.params['backup_date']
        return [
            "select date_format(max(last_update),'%Y%m%d') >= {} from adfox.campaign".format(date),
            "select date_format(max(last_update),'%Y%m%d') >= {} from adfox.banner".format(date),
            "select date_format(max(last_update),'%Y%m%d') >= {} from adfox.campaign_site".format(date),
            "select date_format(max(last_update),'%Y%m%d') >= {} from adfox.campaign_zone".format(date),
            "select date_format(max(last_update),'%Y%m%d') >= {} from adfox.campaign_place".format(date),
        ]

    def check_last_update(self):
        for query in self.get_last_update_queries():
            logging.info('checking if is true: {}'.format(query))
            cur = self.get_db_conn().cursor()
            cur.execute(query)
            res = cur.fetchall()
            cur.close()
            if not res[0][0]:
                raise LastUpdateCheckError

    @staticmethod
    def get_master_config():
        return {
            'host': os.environ['MYSQL_HOST'],
            'user': os.environ['MYSQL_USER'],
            'password': os.environ['MYSQL_PASSWORD'],
        }

    def proceed(self):
        self.create_slave()
        self.check_last_update()
        for i in range(0, int(self.params['slave_wait'])):
            if not self.check_slave_errors():
                raise Exception
            logging.info('slave check: slave is ok, no errors')
            time.sleep(SLAVE_SLEEP_TIME)


def main():
    logging.getLogger().setLevel(logging.DEBUG)
    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    handler.setFormatter(formatter)
    logging.getLogger().addHandler(handler)
    logging.info('Starting script...')

    parser = ArgumentParser(description='backup validation')
    parser.add_argument('--slave-wait', dest='slave_wait',
                        help="checking slave status for...in minutes", required=True)
    parser.add_argument('--cnf-file', dest='cnf_file', default='/etc/mysql/my.cnf')
    parser.add_argument('--backup-date', dest='backup_date',
                        help="backup date for check in format like '%%Y%%m%%d'", required=True)
    args = parser.parse_args()
    BackupValidation(cnf_file=args.cnf_file, slave_wait=args.slave_wait, backup_date=args.backup_date).proceed()


if __name__ == '__main__':
    main()
