#!/usr/bin/env python
import subprocess
import re
import os
import sys
import logging
import ConfigParser
import boto3
from argparse import ArgumentParser
from datetime import datetime, timedelta


class MdsS3:
    def __init__(self, url, bucket_name, access_key=None, secret_key=None):
        self.url = url
        self.bucket_name = bucket_name
        self.client = boto3.client('s3', endpoint_url=url, aws_access_key_id=access_key,
                                   aws_secret_access_key=secret_key,
                                   use_ssl=False, verify=False)

    def list(self, prefix=''):
        return self.client.list_objects(Bucket=self.bucket_name, Prefix=prefix)

    def download(self, key, f_name):
        self.client.download_file(self.bucket_name, key, f_name)


class InnobackupexError(Exception):
    def __init__(self, output):
        self.output = output


class BackupNotFound(Exception):
    pass


def get_run_parameters():
    parser = ArgumentParser(description='Xtrabackup restore tool')

    parser.add_argument('--url', dest='url', help="s3 server url", required=True)
    parser.add_argument('--bucket-name', dest='bucket_name', help="s3 bucket name", required=True)
    parser.add_argument('--last', dest='last', action='store_true', help='Take any yesterday backup')
    parser.add_argument('--name', dest='name', help="full name of mds file like '%%Y%%m%%d....tar.xz'")
    parser.add_argument('--backup-host', dest='backup_host',
                        help="backup host in format like 'bkp1.myt.adfox.ru'", required=True)
    parser.add_argument('--backup-date', dest='backup_date',
                        help="backup date for check in format like '%%Y%%m%%d'", required=True)
    parser.add_argument('--keep-files', dest='keep_files', action='store_true',
                        help='Don\'t delete backup files after restore', default=False)
    parser.add_argument('--slave-info', dest='slave_info', action='store_true', help='Print slave info')

    parser.add_argument('--tmp_dir', dest='tmp_dir',
                        help='tmp dir for backup download and apply log', default='/tmp')
    parser.add_argument('--cnf-file', dest='cnf_file', help='location of mysql config', default='/etc/mysql/my.cnf')
    parser.add_argument('--innodb-memory', dest='innodb_memory', help="use memory for innobackupex", default='8G')
    return parser.parse_args()


def get_backup_name(args):
    if not args.name and not args.last and not (args.backup_host and args.backup_date):
        raise BackupNotFound
    if args.last:
        yesterday = datetime.today() - timedelta(days=1)
        date = yesterday.strftime('%Y%m%d')
        mds = MdsS3(url=args.url, bucket_name=args.bucket_name)
        ls_res = mds.list('mysql/{}'.format(date))
        return ls_res['Contents'][0]['Key']
    elif args.backup_host and args.backup_date:
        date = args.backup_date
        mds = MdsS3(url=args.url, bucket_name=args.bucket_name)
        ls_res = mds.list('mysql/{}'.format(date))
        pattern = re.compile(args.backup_host)
        for keys in ls_res['Contents']:
            if re.search(pattern, keys['Key']):
                return keys['Key']
        raise BackupNotFound
    else:
        return 'mysql/{}'.format(args.name)


def main():
    logging.getLogger().setLevel(logging.DEBUG)
    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(logging.DEBUG)
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
    handler.setFormatter(formatter)
    logger = logging.getLogger()
    logger.addHandler(handler)
    logging.getLogger('boto3').setLevel(logging.CRITICAL)
    logging.getLogger('botocore').setLevel(logging.CRITICAL)
    logging.getLogger('s3transfer').setLevel(logging.CRITICAL)
    logging.getLogger('urllib3').setLevel(logging.CRITICAL)
    logging.info(' Starting Xtrabackup...')
    args = get_run_parameters()
    tmp_name = tmp_dir = None
    try:
        name = get_backup_name(args)
        logging.info('downloading backup {}'.format(name))
        tmp_add_dir = datetime.now().strftime("%Y%m%d%H%M%S")
        tmp_dir_base = args.tmp_dir
        tmp_dir = os.path.join(tmp_dir_base, tmp_add_dir)
        if not os.path.isdir(tmp_dir):
            os.makedirs(tmp_dir)

        tmp_name = '{}/{}'.format(tmp_dir, re.sub('^mysql/', '', name))
        mds = MdsS3(url=args.url, bucket_name=args.bucket_name)
        mds.download(name, tmp_name)
        logging.info('starting tar extracting')
        subprocess.call('cd {} && tar xf {}'.format(tmp_dir, tmp_name), shell=True)
        only_dirs = [f for f in os.listdir(tmp_dir) if os.path.isdir(os.path.join(tmp_dir, f))]
        if len(only_dirs) != 1:
            raise Exception
        backup_abs_path = os.path.join(tmp_dir, only_dirs[0])
        subprocess.call('innobackupex --apply-log --use-memory={} {}'
                        .format(args.innodb_memory, backup_abs_path), shell=True, stderr=subprocess.STDOUT)
        config = ConfigParser.ConfigParser(allow_no_value=True)
        config.read(args.cnf_file)
        datadir = config.get('mysqld', 'datadir')
        if not os.path.isdir(datadir):
            os.makedirs(datadir)
        try:
            innodb_log_dir = config.get('mysqld', 'innodb_log_group_home_dir')
            if not os.path.isdir(innodb_log_dir):
                os.makedirs(innodb_log_dir)
        except ConfigParser.NoOptionError:
            innodb_log_dir = datadir
        subprocess.call('service mysql stop', shell=True)
        subprocess.call('rm -rf {}/*'.format(datadir), shell=True)
        if innodb_log_dir != datadir:
            subprocess.call('rm -rf {}/*'.format(innodb_log_dir), shell=True)
        logging.info('running innobackupex --copy-back  {}'.format(backup_abs_path))
        subprocess.call(' innobackupex --copy-back  {}'.format(backup_abs_path),
                        shell=True, stderr=subprocess.STDOUT)
        logging.info('starting mysql...')
        subprocess.call('chown -R mysql:mysql {}/'.format(datadir), shell=True)
        subprocess.call('chown -R mysql:mysql {}/'.format(innodb_log_dir), shell=True)
        subprocess.call('service mysql start', shell=True)
        logging.info('xtrabackupfinished')
    except Exception:
        raise
    finally:
        if not args.keep_files:
            if tmp_name is not None and os.path.exists(tmp_name):
                subprocess.call('rm {}'.format(tmp_name), shell=True)
            if tmp_dir is not None and os.path.exists(tmp_dir):
                subprocess.call('rm -rf {}'.format(tmp_dir), shell=True)


if __name__ == '__main__':
    main()
