# -*- coding: utf-8 -*-

import json
from os.path import join as join_path, realpath, dirname

from sandbox.sandboxsdk import environments
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.juggler import jclient


class AdfoxChUpdateDailyCounters(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        object = sdk2.parameters.String('database object', required=True)
        day_to_recount = sdk2.parameters.String('day to recount', required=True)
        force_update = sdk2.parameters.Bool(
            'force update day (CAUTION: IT WILL IGNORE ALL VALIDATIONS AND CAN MAKE COUNTERS NON-CONSISTENT',
            default=False, required=True)
        juggler_host = sdk2.parameters.String('juggler host', default=None, required=False)

        with sdk2.parameters.Group("rtd") as rtd_params:
            rtd_host = sdk2.parameters.String('RTD host', default='http://rtd.adfox.ru/rpc', required=True)
            rtd_max_wait_result = sdk2.parameters.Integer('Max wait for rtd result', default=180, required=True)
            rtd_repeat_check = sdk2.parameters.Integer('RTD wait for repeat check', default=1, required=True)

        with sdk2.parameters.Group("mysql") as db_params:
            mysql_host = sdk2.parameters.String('Database host', default='db2.fin.adfox.ru', required=True)
            mysql_user = sdk2.parameters.String('Database user', default='ro', required=True)
            db_password_vault = sdk2.parameters.String('Vault for password',
                                                       default='ADFOX_HOURLY_UPDATE_COUNTERS_DB_PASS')
            mysql_main_db = sdk2.parameters.String('Main db', default='test', required=True)

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='ADFOX_CH_UPDATE_DAILY_COUNTERS_{}'.format(self.Parameters.object),
                                       weight=1, capacity=1)
            ],
        )

    def on_execute(self):
        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('pandas', venv=venv, use_wheel=True, version='0.23.4').prepare()
            environments.PipEnvironment('pymysql', venv=venv, use_wheel=True, version='0.9.3').prepare()
            environments.PipEnvironment('python-jsonrpc', venv=venv, use_wheel=True, version='0.9.3').prepare()

            script_path = join_path(dirname(realpath(__file__)), 'ch_update_daily_counters.py')

            mysql = {
                'host': self.Parameters.mysql_host,
                'user': self.Parameters.mysql_user,
                'password': sdk2.Vault.data('ADFOX', self.Parameters.db_password_vault),
                'db': self.Parameters.mysql_main_db,
            }

            rtd = {
                'host': self.Parameters.rtd_host,
                'max_wait_result': self.Parameters.rtd_max_wait_result,
                'repeat_check': self.Parameters.rtd_repeat_check,
            }

            settings = {
                'object': self.Parameters.object,
                'day_to_recount': self.Parameters.day_to_recount,
                'force_update': self.Parameters.force_update,
            }

            run_process([
                venv.executable,
                script_path,
                "--mysql='{}'".format(json.dumps(mysql)),
                "--rtd='{}'".format(json.dumps(rtd)),
                "--settings='{}'".format(json.dumps(settings)),
            ], log_prefix='ch_update_daily_counters', shell=True)

        if self.Parameters.juggler_host:
            jclient.send_events_to_juggler(
                self.Parameters.juggler_host,
                'ch_update_daily_counters_{}'.format(self.Parameters.object),
                'OK',
                'Script has been successfully completed.'
            )
