#!/usr/bin/env python2.7
import pandas
import pymysql
import logging
import sys
import warnings
from counters_base_class import CountersBase
from datetime import datetime


class RecountDayError(Exception):
    pass


class ChUpdateDailyCounters(CountersBase):
    def __init__(self, **kwargs):
        super(ChUpdateDailyCounters, self).__init__(**kwargs)
        self.params = kwargs
        self.day_to_recount = datetime.strptime(self.params['settings']['day_to_recount'], '%Y-%m-%d')

    def validate_day_to_recount(self):
        conn = self.get_db_conn()
        query = "select min(day) as min_day, max(day) as max_day from {table}".format(table=self.get_db_daily_table())
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        cursor.execute(query)
        result = cursor.fetchone()
        cursor.close()
        conn.close()
        if result['min_day'] > self.day_to_recount.date():
            logging.error('min day in daily table larger then day to recount')
            raise RecountDayError
        if result['max_day'] < self.day_to_recount.date():
            logging.info('max day in daily table lesser then day to recount')
            raise RecountDayError

    def db_update_data(self, data):
        update_data = self.make_update_data_from_ch(data=data, update_hour=False)
        query = """
            INSERT INTO {tablename}
            ({column})
            VALUES
            {values}
            ON DUPLICATE KEY UPDATE
                impressions = VALUES(impressions),
                clicks = VALUES(clicks)
        """.format(tablename=self.get_db_daily_table(),
                   column=",".join(update_data['columns']), values=",".join(update_data['values']))
        conn = self.get_db_conn()
        conn.cursor().execute(self.get_nullify_daily_day_query(self.day_to_recount.strftime("%Y-%m-%d")))
        conn.cursor().execute(query)
        conn.commit()

    def proceed(self):
        self.set_logger()
        try:
            if not self.params['settings']['force_update']:
                self.validate_day_to_recount()
            logging.info('recounting data from ch for {} day for {} object'
                         .format(self.day_to_recount.strftime("%Y-%m-%d"), self.object))
            res = self.get_ch_data(day=self.day_to_recount.strftime("%Y-%m-%d"))
            data = pandas.DataFrame(res['table'], columns=res['fields'])\
                .query('(impressions_total > 0 or clicks_total > 0) and {}_id > 0'
                       .format(self.object_primary))
            logging.info('Got ch data')
            self.db_update_data(data=data.to_dict('records'))
            logging.info('update successfully done')
            logging.info('Script has been successfully completed. ')
            return True
        except Exception:
            raise
        finally:
            self.close_db_conn()


if __name__ == "__main__":
    if len(sys.argv) == 1:
        # execute from command line with no args: let's use config file for settings
        import config as conf
        params = conf.params

    else:
        # args not empty: let's take settings from args as json.dumps dicts
        params = ChUpdateDailyCounters.parse_args('mysql', 'rtd', 'settings')
    ChUpdateDailyCounters(**params).proceed()
