#!/usr/bin/env python2.7
import pandas
import pymysql
import logging
import sys
import time
import warnings
from counters_base_class import CountersBase, CHHorizonError
from datetime import datetime, timedelta

MIN_CH_HORIZON_DELTA_IN_HOURS = 1
MIN_HOURS_UPDATE = 1


class ChUpdateHourlyCounters(CountersBase):
    def __init__(self, **kwargs):
        super(ChUpdateHourlyCounters, self).__init__(**kwargs)
        self.params = kwargs
        self.day_to_recount = None
        self.hours_to_recount = None

    def get_hours_for_recount(self):
        conn = self.get_db_conn()
        # lets find min hour witch has no flag_ch_updated rows
        query = "select date_add(max(time), interval 1 hour) from {table} where find_in_set('flag_ch_updated', flags)"\
            .format(table=self.get_db_hourly_table())
        # if there is no any flag_ch_updated rows - proceed from min time
        query_if_none = "select min(time) from {table}".format(table=self.get_db_hourly_table())
        with conn as cursor:
            cursor.execute(query)
            result = cursor.fetchall()
            if result[0][0] is None:
                cursor.execute(query_if_none)
                result = cursor.fetchall()
        conn.close()
        db_min_not_updated_datetime = result[0][0]
        logging.info('min not updated hour is {}'.format(db_min_not_updated_datetime))
        event_horizon = self.get_event_horizon()
        # subtract some hours from event horizon to make reserved window for ch if horizon is not precise
        subtract_hours_from_ch_horizon = self.params['settings']['subtract_hours_from_ch_horizon'] \
            if self.params['settings']['subtract_hours_from_ch_horizon'] >= MIN_CH_HORIZON_DELTA_IN_HOURS \
            else MIN_CH_HORIZON_DELTA_IN_HOURS
        event_horizon = event_horizon - timedelta(hours=subtract_hours_from_ch_horizon)
        if event_horizon.strftime("%Y-%m-%d") > db_min_not_updated_datetime.strftime("%Y-%m-%d"):
            hour = list(range(int(db_min_not_updated_datetime.strftime("%H")), 24))
        elif event_horizon.strftime("%Y-%m-%d") == db_min_not_updated_datetime.strftime("%Y-%m-%d"):
            hour = list(range(int(db_min_not_updated_datetime.strftime("%H")),
                              int(event_horizon.strftime("%H"))))
        else:
            raise CHHorizonError
        max_hours_update = self.params['settings']['max_hours_update'] \
            if self.params['settings']['max_hours_update'] >= MIN_HOURS_UPDATE else MIN_HOURS_UPDATE
        if len(hour) > max_hours_update:
            hour = hour[0:max_hours_update]
        if len(hour) == 0:
            logging.info('nothing to recount')
            exit(0)
        self.day_to_recount = db_min_not_updated_datetime.strftime("%Y-%m-%d")
        self.hours_to_recount = hour
        logging.info('updating day: {}'.format(self.day_to_recount))
        logging.info('updating hours: {}'.format(hour))

    def db_update_data(self, data, hour):
        update_data = self.make_update_data_from_ch(data=data, update_hour=True)
        query = """
            INSERT INTO {tablename}
            ({column})
            VALUES
            {values}
            ON DUPLICATE KEY UPDATE
                impressions = VALUES(impressions),
                clicks = VALUES(clicks)
        """.format(tablename=self.get_db_hourly_table(),
                   column=",".join(update_data['columns']), values=",".join(update_data['values']))
        self.db_make_update(query=query, hour=hour)

    @CountersBase.transaction_repeat_decorator
    def db_make_update(self, query, hour):
        conn = self.get_db_conn()
        conn.cursor().execute(self.get_nullify_hourly_hour_query(hour))
        conn.cursor().execute(query)
        conn.commit()

    def proceed(self):
        # ignore sql warnings: concat_ws on set fields
        warnings.simplefilter("ignore", category=pymysql.Warning)
        self.set_logger()
        try:
            self.get_hours_for_recount()
            res = self.get_ch_data(day=self.day_to_recount, hour=self.hours_to_recount)
            data = pandas.DataFrame(res['table'], columns=res['fields'])\
                .query('(impressions_total > 0 or clicks_total > 0) and {}_id > 0'
                       .format(self.object_primary))
            logging.info('Got ch data')
            if (data.correct == 1).all():
                hours = [(datetime.strptime(self.day_to_recount, "%Y-%m-%d")
                          + timedelta(hours=h)) for h in self.hours_to_recount]
                for h in hours:
                    data_h = data.query('date_hour==\'{}\''.format(h))
                    self.db_update_data(data=data_h.to_dict('records'), hour=h)
                    logging.info('update successfully done for {} hour'.format(h))
            else:
                logging.info('Update stopped. Not all data is correct (correct != 1)')
            logging.info('Script has been successfully completed. ')
            return True
        except Exception:
            raise
        finally:
            self.close_db_conn()


if __name__ == "__main__":
    if len(sys.argv) == 1:
        # execute from command line with no args: let's use config file for settings
        import config as conf
        params = conf.params

    else:
        # args not empty: let's take settings from args as json.dumps dicts
        params = ChUpdateHourlyCounters.parse_args('mysql', 'rtd', 'settings')
    ChUpdateHourlyCounters(**params).proceed()
