#!/usr/bin/env python2.7
import logging
import sys
import pymysql
import warnings
from counters_base_class import CountersBase
from datetime import timedelta

MIN_KEEP_DAILY_DAYS = 1


class MergeDailyCountersIntoTotal(CountersBase):
    def __init__(self, **kwargs):
        super(MergeDailyCountersIntoTotal, self).__init__(**kwargs)
        self.params = kwargs
        self.days_to_merge = None
        self.max_days_to_merge = self.params['settings']['max_days_to_merge']
        self.keep_daily_days = self.params['settings']['keep_daily_days']  \
            if self.params['settings']['keep_daily_days'] >= MIN_KEEP_DAILY_DAYS else MIN_KEEP_DAILY_DAYS

    def get_days_to_merge(self):
        conn = self.get_db_conn()
        # lets find min non-merged day
        query = "select timestamp(min(day)) as min_day, timestamp(max(day)) as max_day from {table} " \
                "where not find_in_set('flag_merged', flags)".format(table=self.get_db_daily_table())
        cursor = conn.cursor(pymysql.cursors.DictCursor)
        cursor.execute(query)
        result = cursor.fetchone()
        cursor.close()
        conn.close()
        min_day = result['min_day']
        max_day = result['max_day'] - timedelta(days=self.keep_daily_days) + timedelta(days=1)
        if max_day > min_day + timedelta(days=self.max_days_to_merge):
            max_day = min_day + timedelta(days=self.max_days_to_merge)
        # real max_day is max_day minus 1 day: it will counted in range() below
        if max_day <= min_day:
            logging.info('nothing to merge')
        else:
            self.days_to_merge = [min_day + timedelta(days=x) for x in range(0, (max_day - min_day).days)]
            logging.info('days to merge: {}'.format(', '.join([d.strftime("%Y-%m-%d") for d in self.days_to_merge])))

    def get_merge_query(self, day):
        query = """
            INSERT INTO {total_table}
            ({id_fields}, impressions, clicks)
            SELECT {id_fields}, sum(impressions), sum(clicks)
            FROM {daily_table}
            WHERE day = '{day}'
            GROUP BY {id_fields}
            ON DUPLICATE KEY UPDATE
            impressions = impressions + VALUES(impressions), clicks = clicks + VALUES(clicks)
        """.format(total_table=self.get_db_total_table(), daily_table=self.get_db_daily_table(),
                   id_fields=','.join(self.get_id_fields()), day=day)
        return query

    def db_merge_single_day(self, day):
        conn = self.get_db_conn()
        logging.info('merging data in total table for {} day'.format(day))
        conn.cursor().execute(self.get_merge_query(day))
        logging.info('merging is finished')
        logging.info('marking daily table data merged for {} day'.format(day))
        affected_rows = conn.cursor().execute(self.get_daily_mark_merged_day_query(day))
        logging.info('marking is done for {} rows'.format(affected_rows))
        conn.commit()
        logging.info('changes committed')

    def remove_old_data_from_daily_table(self):
        conn = self.get_db_conn()
        logging.info('removing old data from daily table')
        affected_rows = conn.cursor().execute(self.get_remove_old_data_from_daily_table_query())
        logging.info('removed {} rows from daily table'.format(affected_rows))
        conn.commit()
        logging.info('changes commited')

    def proceed(self):
        # ignore sql warnings: concat_ws on set fields
        warnings.simplefilter("ignore", category=pymysql.Warning)
        self.set_logger()
        try:
            self.get_days_to_merge()
            if self.days_to_merge:
                for day in self.days_to_merge:
                    self.db_merge_single_day(day=day.date())
            self.remove_old_data_from_daily_table()
            logging.info('Script has been successfully completed. ')
            return True
        except Exception:
            raise
        finally:
            self.close_db_conn()


if __name__ == "__main__":
    if len(sys.argv) == 1:
        # execute from command line with no args: let's use config file for settings
        import config as conf
        params = conf.params

    else:
        # args not empty: let's take settings from args as json.dumps dicts
        params = MergeDailyCountersIntoTotal.parse_args('settings', 'mysql')
    MergeDailyCountersIntoTotal(**params).proceed()
