# -*- coding: utf-8 -*-

import json
from os.path import join as join_path, realpath, dirname

from sandbox.sandboxsdk import environments
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.juggler import jclient


class AdfoxMergeHourlyCountersIntoDaily(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        object = sdk2.parameters.String('database object', required=True)
        keep_hourly_days = sdk2.parameters.Integer('keep full days in hourly counters table', default=2, required=True)
        max_days_to_merge = sdk2.parameters.Integer('max days to merge', default=7, required=True)
        delete_from_daily_table = sdk2.parameters.Bool('delete data from daily table before insert',
                                                       default=False, required=True)
        juggler_host = sdk2.parameters.String('juggler host', default=None, required=False)

        with sdk2.parameters.Group("mysql") as db_params:
            mysql_host = sdk2.parameters.String('Database host', default='db2.fin.adfox.ru', required=True)
            mysql_user = sdk2.parameters.String('Database user', default='ro', required=True)
            db_password_vault = sdk2.parameters.String('Vault for password',
                                                       default='ADFOX_HOURLY_UPDATE_COUNTERS_DB_PASS')
            mysql_main_db = sdk2.parameters.String('Main db', default='test', required=True)

    def on_enqueue(self):
        self.Requirements.semaphores = ctt.Semaphores(
            acquires=[
                ctt.Semaphores.Acquire(name='ADFOX_MERGE_HOURLY_COUNTERS_INTO_DAILY_{}'.format(self.Parameters.object),
                                       weight=1, capacity=1)
            ],
        )

    def on_execute(self):
        with environments.VirtualEnvironment() as venv:
            environments.PipEnvironment('pymysql', venv=venv, use_wheel=True, version='0.9.3').prepare()

            script_path = join_path(dirname(realpath(__file__)), 'merge_hourly_counters_into_daily.py')

            mysql = {
                'host': self.Parameters.mysql_host,
                'user': self.Parameters.mysql_user,
                'password': sdk2.Vault.data('ADFOX', self.Parameters.db_password_vault),
                'db': self.Parameters.mysql_main_db,
            }

            settings = {
                'object': self.Parameters.object,
                'keep_hourly_days': self.Parameters.keep_hourly_days,
                'max_days_to_merge': self.Parameters.max_days_to_merge,
                'delete_from_daily_table': self.Parameters.delete_from_daily_table,
            }

            run_process([
                venv.executable,
                script_path,
                "--mysql='{}'".format(json.dumps(mysql)),
                "--settings='{}'".format(json.dumps(settings)),
            ], log_prefix='merge_hourly_counters_into_daily', shell=True)

        if self.Parameters.juggler_host:
            jclient.send_events_to_juggler(
                self.Parameters.juggler_host,
                'merge_hourly_counters_into_daily_{}'.format(self.Parameters.object),
                'OK',
                'Script has been successfully completed.'
            )
