# -*- coding: utf-8 -*-

import logging
import socket
import xmlrpclib
from datetime import datetime

from sandbox import common
from sandbox.sandboxsdk import environments

from sandbox import sdk2

CURRENCY_DICT = dict(AUD=36, AZN=944, AMD=51, BYN=933, BGN=975, BRL=986, HUF=348, KRW=410, HKD=344,
                     DKK=208, USD=840, EUR=978, INR=356, KZT=398, CAD=124, KGS=417, CNY=156, MDL=498,
                     TMT=934, NOK=578, PLN=985, RON=946, XDR=960, SGD=702, TJS=972, TRY=949, UZS=860,
                     UAH=980, GBP=826, CZK=203, SEK=752, CHF=756, ZAR=710, JPY=392, RUB=643)

logging.basicConfig(level=logging.DEBUG)
logger = logging.getLogger('ADFOX_GET_CURRENCY_RATES')
logger.setLevel(logging.DEBUG)

BALANCE_URLS = dict(TEST='http://greed-tm1h.paysys.yandex.net:8002/xmlrpc/',
                    PROD='http://balance-xmlrpc.yandex.net:8002/xmlrpc/')
DATABASE_URLS = dict(TEST='mysql://{}@devel-login.adfox.yandex-team.ru:3306/devel',
                     PROD='mysql://{}@db.adfox.ru:3306/adfox')


def make_error(error_string):
    logger.debug(error_string)
    raise common.errors.TaskFailure(error_string)


class AdfoxGetCurrencyRates(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):

        currencies = sdk2.parameters.List('Currency', default=['USD', 'EUR'], required=True)
        with sdk2.parameters.RadioGroup('Balance service type', required=True) as balance:
            balance.values['TEST'] = balance.Value('TEST', default=True)
            balance.values['PROD'] = 'PROD'
        with sdk2.parameters.RadioGroup('Destination database type', required=True) as database:
            database.values['TEST'] = database.Value('TEST', default=True)
            database.values['PROD'] = 'PROD'
        db_user = sdk2.parameters.String('User to connect to database', default='sandbox-uploader', required=True)
        db_password_vault = sdk2.parameters.String('Vault for password', default='ADFOX_MYSQL_SANDBOX_UPLOADER_PASSWORD')

        tablename = sdk2.parameters.String('Tablename in destination database', default='currency_rates', required=True)
        timeout = sdk2.parameters.Integer('Balance service response waiting time, ms', default=1000, required=True)
        rep_num = sdk2.parameters.Integer(
            'Number of tries to get response from balance service', default=3, required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = (
            environments.PipEnvironment('mysqlclient', version="1.4.6"),
            environments.PipEnvironment('sqlalchemy', version="1.3.23"),
        )

    def get_login_credentials(self):
        return '{user}:{pswrd}'.format(
            user=self.Parameters.db_user,
            pswrd=sdk2.Vault.data('ADFOX', self.Parameters.db_password_vault),
        )

    def on_execute(self):
        from sqlalchemy import Column, Integer, Date, PrimaryKeyConstraint, create_engine
        from sqlalchemy.ext.declarative import declarative_base
        from sqlalchemy.orm import sessionmaker

        logger.debug('Checking task parameters...')
        currencies = self.Parameters.currencies
        if not currencies:
            make_error('Currency field must not be empty')
        for char_code in currencies:
            if char_code not in CURRENCY_DICT.keys():
                make_error('Currency char code "{}" is not allowed'.format(char_code))

        logger.debug('Balance service type: {}'.format(self.Parameters.balance))
        logger.debug('Destination database type: {}'.format(self.Parameters.database))

        login = self.get_login_credentials()
        database_url = DATABASE_URLS[self.Parameters.database].format(login)
        balance_url = BALANCE_URLS[self.Parameters.balance]

        logger.debug('Requesting data from {} balance service...'.format(self.Parameters.balance))
        server = xmlrpclib.ServerProxy(balance_url)
        socket.setdefaulttimeout(self.Parameters.timeout)
        result_list = []
        rep_num = self.Parameters.rep_num
        for char_code in currencies:
            ntry = 0
            result = dict()
            while ntry < rep_num and not result:
                ntry += 1
                try:
                    res = server.Balance.GetCurrencyRate(char_code, '')
                except Exception as e:
                    logger.debug(
                        'Requesting {} rate... Try {} out of {}. Exception {}'.format(char_code, ntry, rep_num, e))
                else:
                    if res == 'FAULT':
                        logger.debug(
                            'Requesting {} rate... Try {} out of {}. FAULT response: {}'.format(char_code, ntry, rep_num, res)
                        )
                    elif not isinstance(res, list) or len(res) < 2:
                        logger.debug('Requesting {} rate... Try {} out of {}. Wrong response format: {}'.
                                     format(char_code, ntry, rep_num, res))
                    elif res[1] != 'SUCCESS':
                        logger.debug('Requesting {} rate... Try {} out of {}. Unsuccessful response: {}'.
                                     format(char_code, ntry, rep_num, res))
                    else:
                        result = res[2]
                        logger.debug('Requesting {} rate... Try {} out of {}. Successful response: {}'.
                                     format(char_code, ntry, rep_num, res))
            if result:
                result_list.append(result)
            else:
                logger.debug('{} rate was NOT obtained'.format(char_code))
        socket.setdefaulttimeout(None)
        logger.debug('From balance service: {}'.format(result_list))

        if not result_list:
            make_error('No data obtained from balance service')

        logger.debug('Checking obtained data...')
        for result in result_list:
            char_code = result.get('currency')
            rate_date = datetime.strptime(result['date'].value, '%Y%m%dT%H:%M:%S').date()
            today = datetime.now().date()
            if char_code not in currencies:
                logger.debug('Provided currency code {} was not asked'.format(char_code))
                result_list.pop(result)
            elif rate_date != today:
                logger.debug('Provided date of {} rate {} is not equal to today\'s date {}'.format(char_code, rate_date, today))
                result_list.pop(result)

        logger.debug('Transforming data format...')
        for result in result_list:
            result['currency_id'] = CURRENCY_DICT[result.pop('currency')]
            result['rate_date'] = datetime.strptime(result.pop('date').value, '%Y%m%dT%H:%M:%S').date()
            rate_parts = result['rate'].split('.')
            result['rate'] = ''.join(rate_parts)
            result['divider'] = 1 if len(rate_parts) == 1 else 10 ** (len(rate_parts[-1]))
        logger.debug('To Database: {}'.format(result_list))

        class CurrencyRates(declarative_base()):
            __tablename__ = self.Parameters.tablename
            __table_args__ = (PrimaryKeyConstraint('currency_id', 'rate_date'),)
            currency_id = Column(Integer)
            rate_date = Column(Date)
            rate = Column(Integer)
            divider = Column(Integer)

            def __init__(self, currency_id=643, rate_date=datetime.now().date(), rate=1, divider=1):
                self.currency_id = currency_id
                self.rate_date = rate_date
                self.rate = rate
                self.divider = divider

        logger.debug('Writing data to {} database...'.format(self.Parameters.database))
        engine = create_engine(database_url, connect_args={'charset': 'utf8'}, echo=False)
        Session = sessionmaker(bind=engine)
        session = Session()
        try:
            for result in result_list:
                session.merge(CurrencyRates(**result))
            session.commit()
            logger.debug('Data was successfully written to database')
        except Exception as e:
            session.rollback()
            make_error('Connection to database was unsuccessful. Exception {}'.format(e))

        ISO_DICT = {v: k for k, v in CURRENCY_DICT.items()}
        not_obtained_currencies = set(currencies)-set([ISO_DICT[result['currency_id']] for result in result_list])
        if not_obtained_currencies:
            make_error('{} rates were NOT obtained'.format(', '.join(list(not_obtained_currencies))))
