# -*- coding: utf-8 -*-
import os
import logging
import sys
import datetime
import sandbox.common.types.task as ctt
from shutil import copyfile
from os.path import join as join_path, realpath, dirname
from sandbox.projects.adfox.adfox_ui.util.config import JugglerConfig
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.juggler import jclient
from sandbox import sdk2

# https://sandbox.yandex-team.ru/resource/2175643453/view
LXC_CONTAINER_ID = 2175643453
TMP_DIR = '/tmp'
MY_DIR = '/etc/mysql'
MY_CNF = 'my.cnf'
MYSQL_SLAVE_CONFIG_DIR = '.mysql'
MYSQL_SLAVE_CONFIG_FILE = 'config'
DEFAULT_JUGGLER_SERVICE = 'adfox-mdb-mysql-backup-validation'


class AdfoxMdbBackupValidate(sdk2.Task):

    class Requirements(sdk2.Requirements):
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        _container = sdk2.parameters.Container(
            'LXC Container with mysql-5.7',
            default_value=LXC_CONTAINER_ID,
            platform='linux_ubuntu_16.04_xenial',
            required=True,
        )
        url = sdk2.parameters.String('s3 mds url', default='http://s3.mds.yandex.net', required=True)
        bucket_name = sdk2.parameters.String('s3 bucket name', default='adfox', required=True)
        check_today = sdk2.parameters.Bool("Check today's backup")
        with check_today.value[False]:
            backup_date = sdk2.parameters.String("backup date in '%Y%m%d' format", default='20210620', required=True)

        backup_number = sdk2.parameters.String("backup number", default=1, required=True)
        tmp_dir = sdk2.parameters.String('tmp dir', default=TMP_DIR, required=False)
        with sdk2.parameters.Group("aws credentials") as aws_credentials:
            s3_id_vault = sdk2.parameters.String('Vault for s3 id',
                                                 default='ADFOX_S3_ACCESS_KEY_ID')
            s3_secret_vault = sdk2.parameters.String('Vault for s3 secret',
                                                     default='ADFOX_S3_ACCESS_KEY_SECRET')

        with sdk2.parameters.Group("Juggler Parameters") as juggler_block:
            juggler_host = sdk2.parameters.String('Juggler host',
                                                  default=JugglerConfig.DEFAULT_JUGGLER_HOST,
                                                  required=True)
            juggler_service = sdk2.parameters.String(
                'Juggler service',
                default=DEFAULT_JUGGLER_SERVICE,
                required=True
            )

    def on_execute(self):
        logging.getLogger().setLevel(logging.DEBUG)
        handler = logging.StreamHandler(sys.stdout)
        handler.setLevel(logging.DEBUG)
        formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')
        handler.setFormatter(formatter)
        logging.getLogger().addHandler(handler)
        logging.info('Starting script...')
        my_cfn_path = join_path(dirname(realpath(__file__)), MY_CNF)
        my_cfn_dest = join_path(MY_DIR, MY_CNF)
        copyfile(my_cfn_path, my_cfn_dest)
        validate_backup_script_path = join_path(dirname(realpath(__file__)), 'validate_backup.py')
        validate_backup_env = dict(os.environ)
        validate_backup_env["AWS_ACCESS_KEY_ID"] = sdk2.Vault.data('ADFOX', self.Parameters.s3_id_vault)
        validate_backup_env["AWS_SECRET_ACCESS_KEY"] = sdk2.Vault.data('ADFOX', self.Parameters.s3_secret_vault)

        tmp_dir = self.Parameters.tmp_dir
        if not os.path.isdir(tmp_dir):
            os.makedirs(tmp_dir)
        if self.Parameters.check_today:
            today = datetime.datetime.today()
            backup_date = today.strftime('%Y%m%d')
        else:
            backup_date = self.Parameters.backup_date

        logging.info('launching mysqldump_restore')
        run_process([
            validate_backup_script_path,
            "--url='{}'".format(self.Parameters.url),
            "--bucket-name='{}'".format(self.Parameters.bucket_name),
            "--backup-number='{}'".format(self.Parameters.backup_number),
            "--backup-date='{}'".format(backup_date),
            "--tmp_dir='{}'".format(tmp_dir),
            "--cnf-file='{}'".format(my_cfn_dest),
        ], log_prefix='validate_backup', environment=validate_backup_env, shell=True)

        logging.info('Sending OK to juggler')
        self.juggler_notify_ok()
        logging.info('all done')

    def on_break(self, prev_status, status):
        # Do not raise CRIT if the task is stopped
        if status != ctt.Status.STOPPED:
            self.juggler_notify_error()

    def on_failure(self, status):
        self.juggler_notify_error()

    def juggler_notify_error(self):
        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            'CRIT',
            'MDB Mysql backup validation failed. More info: https://wiki.yandex-team.ru/adfox/interfaces/bd/'
        )

    def juggler_notify_ok(self):
        jclient.send_events_to_juggler(
            self.Parameters.juggler_host,
            self.Parameters.juggler_service,
            'OK',
            'MDB Mysql backup validation is OK'
        )
