# -*- coding: utf-8 -*-
import logging
import os
from datetime import datetime, timedelta

from sandbox import sdk2
from sandbox.common.types.task import Status
from sandbox.common.utils import chain
from sandbox.sandboxsdk import environments


class AdfoxPredictorSamples(sdk2.Task):
    class Context(sdk2.Task.Context):
        strDateReport = None  # Date, for which to build report
        yql_check_task_id = None

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (
            environments.PipEnvironment('yandex-yt', '0.8.17-0'),
        )

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 16000
        max_restarts = 2

        bl_explicit_date = sdk2.parameters.Bool("Set explicit date to prepare report", default=False, required=True)
        with bl_explicit_date.value[True]:
            dt = datetime.today() - timedelta(days=1)
            date_report = sdk2.parameters.String('Date to build report', default=dt.strftime("%Y-%m-%d"),
                                                 required=False)

        yt_samples_directory = sdk2.parameters.String('YT Samples Directory',
                                                      default='//home/adfox-forecast/v2_samples',
                                                      required=True)
        yt_access_log_directory = sdk2.parameters.String('YT adfox-access-log path',
                                                         default='//logs/adfox-nginx-access-log/1d',
                                                         required=True)
        yt_event_log_directory = sdk2.parameters.String('YT adfox-event-log path', default='//logs/adfox-event-log/1d',
                                                        required=True)

        yt_profiles_directory = sdk2.parameters.String('YT DMP profiles path',
                                                       default='//home/adfox-reports/audience_report',
                                                       required=True)

        with sdk2.parameters.Group("YT Parameters") as yt_block:
            yt_token_vault_owner = sdk2.parameters.String('Vault owner to extract YT token', default='ADFOX',
                                                          required=True)
            yt_cluster = sdk2.parameters.String('YT Cluster', default="HAHN", required=True)

    def __DateReportToStr(self, dateReport):
        return dateReport.strftime("%Y-%m-%d")

    def __GetOwnerGroupsTable(self, strDateReport):
        return self.Parameters.yt_samples_directory + "/owner_groups_" + strDateReport

    def __GetSamplesTable(self, strDateReport):
        return self.Parameters.yt_samples_directory + "/samples_" + strDateReport

    def __GetDmpTable(self, strDateReport):
        return self.Parameters.yt_samples_directory + "/dmp_" + strDateReport

    def __GetDailyAccessLogTable(self, strDateReport):
        return self.Parameters.yt_access_log_directory + "/" + strDateReport

    def __GetDailyEventLogTable(self, strDateReport):
        return self.Parameters.yt_event_log_directory + "/" + strDateReport

    def __GetProfilesTable(self, strDateReport):
        return "{}/{}/profiles".format(self.Parameters.yt_profiles_directory, strDateReport)

    def __ReadTextFile(self, strFileName):
        fullPath = os.path.join(os.path.dirname(os.path.realpath(__file__)), strFileName)
        with open(fullPath, "r") as input_file:
            data = input_file.readlines()
        return "".join(data)

    def __MakeDailyReports(self):
        from projects.yql.RunYQL2 import RunYQL2

        dateReport = datetime.strptime(self.Parameters.date_report,
                                       '%Y-%m-%d').date() if self.Parameters.bl_explicit_date else datetime.today() - timedelta(
            days=1)
        self.Context.strDateReport = dateReport.strftime('%Y-%m-%d')

        query = self.__ReadTextFile('samples_request.sql').format(
            CLUSTER=self.Parameters.yt_cluster,
            OWNER_GROUPS_TABLE=self.__GetOwnerGroupsTable(self.Context.strDateReport),
            SAMPLES_TABLE=self.__GetSamplesTable(self.Context.strDateReport),
            DMP_TABLE=self.__GetDmpTable(self.Context.strDateReport),
            EVENT_LOG_TABLE=self.__GetDailyEventLogTable(self.Context.strDateReport),
            ACCESS_LOG_TABLE=self.__GetDailyAccessLogTable(self.Context.strDateReport),
            PROFILES_TABLE=self.__GetProfilesTable(self.Context.strDateReport),
        )

        query = self.__ReadTextFile('functions.sql') + query

        subtask = RunYQL2(
            self,
            description="Run test YQL query (subtask for {})".format(self.id),
            owner=self.Parameters.yt_token_vault_owner,  # required fields cannot be empty
        )

        subtask.fail_on_any_error = True
        subtask.Parameters.query = query
        subtask.Parameters.publish_query = True
        subtask.Parameters.trace_query = True
        subtask.Parameters.use_v1_syntax = True

        logging.info("Launching query subtask")
        subtask.save()
        subtask.enqueue()

        return subtask

    def on_execute(self):
        logging.info('Running samples generation')

        with self.memoize_stage["run_query"]:
            logging.info('Launching query...')
            self.Context.yql_check_task_id = self.__MakeDailyReports().id
            logging.info("Waiting for subtasks [{}]".format(self.Context.yql_check_task_id))
            raise sdk2.WaitTask(
                [self.Context.yql_check_task_id],
                set(chain(Status.Group.FINISH, Status.Group.BREAK)),
                wait_all=True
            )

        logging.info('Exiting task')
