import datetime as dt
import logging
import os
import shutil
import signal
import tarfile
import time
import uuid
import json

from sandbox.projects.common.yabs.cachedaemon import CacheDaemonStubSandboxNonIntegrated
from sandbox.projects.yabs.qa.module_base import ModuleBase
from sandbox.projects.yabs.qa.server_interface import LocalServerModuleInterface
from sandbox.projects.yabs.qa.sut.utils import reserve_port

from sandbox.projects.adfox.qa.utils import constants
from sandbox.projects.adfox.qa.utils.secrets import get_yav_secret
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess


class AmacsModule(ModuleBase, LocalServerModuleInterface):
    def __init__(self, adapter):
        ModuleBase.__init__(self, adapter)
        self.instance_id = str(uuid.uuid4())

        # ensure required dirs
        self.task_dir = str(self.task.path())
        self.work_dir = os.path.join(self.task_dir, self.instance_id)
        self.logs_dir = os.path.join(get_logs_folder(), self.instance_id)
        self.destinations_dir = os.path.join(self.logs_dir, 'destinations')
        self.queues_dir = os.path.join(self.logs_dir, 'queues')
        self.engine_path = os.path.join(self.work_dir, 'amacs')
        os.mkdir(self.work_dir)
        os.mkdir(self.logs_dir)
        os.mkdir(self.destinations_dir)
        os.mkdir(self.queues_dir)

        self._configuration = dict()
        self._engine_process = None
        self._engine_args = ['--slave', '--daemonize', '--force-cache', '--stderr', '--file',
                             self.adapter.get_atlas_binary_path()]

        # services ports and sockets (sockets aren't used, but should be preserved)
        self._ports = {'listener': 0, 'api_bind': 0}
        self._sockets = []

        # files
        self._prepare_engine_files()
        self._prepare_engine_secrets()
        self._prepare_logs_directories()
        self._prepare_logs_uploader()

        # engine work statistics
        self.work_time = None
        self.user_work_time = None
        self.system_work_time = None
        self.rusage = None

        # cache_daemon
        self._cache_daemon_services = {
            'xpd': ['xpd'],
            'pdb': ['pdb'],
            'rtb': ['rtb'],
            'antifraud': ['antifraud']
        }
        self._cache_daemon = CacheDaemonStubSandboxNonIntegrated(
            cache_daemon_executable_path=self.adapter.get_cache_daemon_executable_path(),
            dump_path=self.adapter.get_cache_daemon_stub_path(),
            data_dir=os.path.join(self.work_dir, 'cache_daemon_data'),
            log_subdir=os.path.join(self.logs_dir, 'cache_daemon_logs'),
            start_on_creation=False, services=self._cache_daemon_services,
            key_header=self.adapter.get_cache_daemon_key_headers())

        self.runs_counter = 0

    def get_process(self):
        return self._engine_process

    def get_port(self):
        return self._ports['listener']

    def get_counters(self):
        import requests
        return requests.get('http://localhost:{0}/counters/all'.format(self.get_port()))

    def __enter__(self):
        self.runs_counter += 1
        self._cache_daemon.__enter__()

        # required on every call
        self._init_ports()  # new ports
        self._prepare_engine_configuration()

        self._engine_log = ProcessLog(logger='engine_log_{0}'.format(self.runs_counter))
        self._engine_log.__enter__()

        # provide task specific configuration via env
        env = {}
        env.update(os.environ)
        env.update({
            "STAGE_TYPE": "SANDBOX_TEST",
            "API_BIND_OPTIONS": "tcp://*:{0}".format(self._configuration["api_bind"]),
            "ANT_SERVER_PORT": self._configuration["listener"],
            "XPD_PORT": self._configuration["xpd"],
            "RTB_PORT": self._configuration["rtb"],
            "PDB_PORT": self._configuration["pdb"],
            "ANTIFRAUD_PORT": self._configuration["antifraud"],
            "CLICKHOUSE_DESTINATIONS_DIR": self._configuration["destinations_dir"],
            "CLICKHOUSE_QUEUES_DIR": self._configuration["queues_dir"],
        })
        env = {k: str(v) for k, v in env.items()}

        self._engine_process = subprocess.Popen([self.engine_path] + self._engine_args,
                                                stdout=self._engine_log.stdout, stderr=self._engine_log.stderr,
                                                cwd=self.work_dir,
                                                env=env)
        ProcessRegistry.register(self._engine_process.pid, [self.engine_path] + self._engine_args)
        self.wait_ping()
        return self

    def wait_ping(self):
        timeout_seconds = 300
        start_dt = dt.datetime.now()
        while dt.datetime.now() - start_dt < dt.timedelta(seconds=timeout_seconds):
            # noinspection PyBroadException
            try:
                self.get_counters()
                return True
            except Exception:
                time.sleep(5)
        raise Exception('Engine did not start in {0} seconds!'.format(timeout_seconds))

    def _init_ports(self):
        self._ports = {'listener': 0, 'api_bind': 0}
        self._sockets = []
        for port_tag in self._ports.iterkeys():
            port, socket = reserve_port()
            self._sockets.append(socket)
            self._ports[port_tag] = int(port)
        self._ports.update(self._cache_daemon.get_ports_by_tag())
        self._configuration.update(self._ports)

    def _prepare_engine_files(self):
        with tarfile.open(self.adapter.get_engine_executable_path()) as tar:
            tar.extractall(path=self.work_dir)
        if os.path.exists(os.path.join(self.work_dir, "adfox/engine/amacs")):
            shutil.move(os.path.join(self.work_dir, "adfox/engine/amacs"), self.work_dir)
            shutil.move(os.path.join(self.work_dir, "adfox/engine/lua"), self.work_dir)

    def _prepare_engine_secrets(self):
        import base64
        secret_data = get_yav_secret(constants.YAV_ROBOT_QA_ADFOX_SHM_KEYS)
        master_key = base64.b64decode(secret_data['master.key'])
        with open(os.path.join(self.work_dir, 'master.key'), 'w') as secret_file:
            secret_file.write(master_key)
        os.makedirs(os.path.join(self.work_dir, 'yauid/keys'))
        with open(os.path.join(self.work_dir, 'yauid/keys', 'master.key'), 'w') as secret_file:
            secret_file.write(master_key)
        os.makedirs(os.path.join(self.work_dir, 'shyid/keys'))

    def _prepare_engine_configuration(self):
        configuration_dir = os.path.join(self.work_dir, 'lua.d')
        if os.path.exists(configuration_dir):
            shutil.rmtree(configuration_dir)
        os.makedirs(configuration_dir)
        with tarfile.open(self.adapter.get_configuration_path()) as tar:
            tar.extractall(path=configuration_dir)

        with open('/tmp/pdb_lookuptable.json', 'w') as pdb_conf_file:
            json.dump(
                {
                    'revision': 1,
                    'slots_count': 1,
                    'endpointmap': {
                        '127.0.0.1:{0}'.format(self._configuration["pdb"]): [0]
                    }
                },
                pdb_conf_file
            )

    def _prepare_logs_directories(self):
        destinations = [
            'logbroker_elog',
            'logbroker_fraud_elog',
            'logbroker_fast_fraud_elog',
            'logbroker_processing_log',
            'logbroker_external_monetizers',
            'logbroker_rtb_log',
            'external_requests_log',
        ]
        for directory in destinations:
            os.makedirs(os.path.join(self.destinations_dir, directory))
        self._configuration.update({
            'queues_dir': self.queues_dir,
            'destinations_dir': self.destinations_dir,
        })

    def _prepare_logs_uploader(self):
        with tarfile.open(self.adapter.get_shoot_logs_uploader_path()) as tar:
            tar.extractall(path=self.work_dir)

    def _save_work_statistics(self, result):
        self.work_time = result[2].ru_utime + result[2].ru_stime
        self.user_work_time = result[2].ru_utime
        self.system_work_time = result[2].ru_stime
        self.rusage = result[2]

    def __exit__(self, *args):
        logging.info('Exiting engine module!')
        logging.info(self.get_counters().text)
        self._cache_daemon.__exit__(*args)

        time.sleep(15)  # sleep to flush all log records
        self._engine_process.send_signal(signal.SIGINT)
        result = os.wait4(self._engine_process.pid, os.WUNTRACED)

        self._save_work_statistics(result)
        self._engine_log.__exit__(*args)

    @property
    def task(self):
        return self.adapter.task_instance

    def upload_logs(self):
        required_logs = [
            ('external_requests_log',
             'result_external_requests_log'),
            ('logbroker_elog',
             'result_event_log'),
            ('logbroker_fraud_elog',
             'result_fraud_event_log'),
            ('logbroker_fast_fraud_elog',
             'result_fast_fraud_event_log'),
            ('logbroker_external_monetizers',
             'result_external_monetizers_log'),
            ('logbroker_processing_log',
             'result_processing_log'),
            ('logbroker_rtb_log',
             'result_rtb_log'),
        ]

        env = os.environ.copy()
        env.update({'YT_TOKEN': get_yav_secret(constants.YAV_ROBOT_QA_ADFOX_YT_TOKEN)['YT_TOKEN']})

        for destination, context_attr in required_logs:
            table_path = self.task.yt.ypath_join(self.task.Context.result_root_directory, destination)
            setattr(self.task.Context, context_attr, table_path)

            # uploading log
            process_args = [
                '--yt-proxy', constants.YT_PROXY,
                '--yt-table-path', table_path,
                '--log-directory', os.path.join(self.destinations_dir, destination),
                '--log-tag', destination,
                '--threads-count', '16',
                '--ttl-days', '100'
            ]

            with ProcessLog(logger='upload_{tag}'.format(tag=destination)) as log:
                subprocess.check_call([os.path.join(self.work_dir, 'shoot_logs_uploader')] + process_args,
                                      stdout=log.stdout, stderr=log.stderr, env=env)
