import datetime as dt
import logging

from sandbox import sdk2

from sandbox.projects.adfox.qa.tasks.AdfoxServerDeleteShootReports.pgaas import select_old_reports, \
    delete_old_reports_records, drop_old_report_table
from sandbox.projects.adfox.qa.utils import constants, pgaas
from sandbox.projects.adfox.qa.tasks.AdfoxServerBase import AdfoxServerBase


class AdfoxServerDeleteShootReports(AdfoxServerBase):
    name = 'ADFOX_SERVER_DELETE_SHOOT_REPORTS'

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('PG configuration') as pgaas_configuration:
            pg_auto_ttl = sdk2.parameters.Integer('TTL for auto reports in PG',
                                                  default_value=constants.AUTO_REPORT_TTL)
            pg_user_ttl = sdk2.parameters.Integer('TTL for user reports in PG',
                                                  default_value=constants.USER_RUN_REPORT_TTL)

        with sdk2.parameters.Group('YT configuration') as yt_configuration:
            yt_auto_ttl = sdk2.parameters.Integer('TTL for auto reports in YT',
                                                  default_value=constants.AUTO_REPORT_TTL)
            yt_user_ttl = sdk2.parameters.Integer('TTL for user reports in YT',
                                                  default_value=constants.USER_RUN_REPORT_TTL)

    def process_pg(self):
        with pgaas.get_connection() as connection:
            with connection.cursor() as cursor:
                old_reports = select_old_reports(cursor,
                                                 self.Parameters.pg_auto_ttl,
                                                 self.Parameters.pg_user_ttl)
                for report in old_reports:
                    drop_old_report_table(cursor, table_name=report['table'])

                logging.info(str(old_reports))
                delete_old_reports_records(cursor, ids_to_drop=[report['id'] for report in old_reports])
                self.set_info('Deleted PG reports: {0}'.format([report['table'] for report in old_reports]))

    @staticmethod
    def get_modification_dt(attr):
        # modification_time example: 2019-06-25T07:29:37.849455Z
        return dt.datetime.strptime(attr.split('.')[0], '%Y-%m-%dT%H:%M:%S')

    def process_yt(self):
        old_reports = []
        nodes = self.yt.list(constants.YT_REPORTS_DIR)
        for node in nodes:
            node_path = self.yt.ypath_join(constants.YT_REPORTS_DIR, node)
            is_auto_run = self.yt.get_attribute(node_path, constants.YT_REPORT_IS_AUTO_RUN_ATTR, default=False)
            ttl = self.Parameters.yt_auto_ttl if is_auto_run else self.Parameters.yt_user_ttl
            modification_dt = self.get_modification_dt(self.yt.get_attribute(node_path, 'modification_time'))
            if dt.datetime.now() - dt.timedelta(days=int(ttl)) > modification_dt:
                old_reports.append(node_path)

        for report in old_reports:
            self.yt.remove(report, recursive=True)
        self.set_info('Deleted YT reports: {0}'.format(old_reports))

    def on_execute(self):
        self.process_pg()
        self.process_yt()
