# -*- coding: utf-8 -*-
import json
import os
import tarfile

from sandbox import sdk2
from sandbox.projects.adfox.qa.resource_types import AdfoxShmShootComparator
from sandbox.projects.adfox.qa.tasks.AdfoxServerBase import AdfoxServerBase
from sandbox.projects.adfox.qa.tasks.AdfoxServerFunctionalShoot import AdfoxServerFunctionalShoot
from sandbox.projects.adfox.qa.tasks.AdfoxServerFunctionalShootCompare.pgaas import create_diff_table, fill_diff_table
from sandbox.projects.adfox.qa.utils import constants, pgaas, retry
from sandbox.projects.adfox.qa.utils.secrets import get_yav_secret
from sandbox.sdk2.helpers import ProcessLog, ProcessRegistry, subprocess

DEFAULT_RULES = {
    'shoot.response.headers.expires': '.*',
    'shoot.response.headers.date': '.*',
    'shoot.response.headers.last-modified': '.*',
    'shoot.response.headers.set-cookie': 'expires=.* GMT',
    'event_log.engine_version': '<DELETE>',
    'fraud_event_log.engine_version': '<DELETE>',
    'fast_fraud_event_log.engine_version': '<DELETE>'
}


class ComparisonParameters(sdk2.Task.Parameters):
    comparator_executable = sdk2.parameters.LastReleasedResource(
        'Comparator executable that performs MapReduce operations',
        required=True, resource_type=AdfoxShmShootComparator)
    save_results = sdk2.parameters.Bool('Save report to PG', required=True, default=True)
    default_rules = sdk2.parameters.JSON('Default rules for diff hiding', required=True, default=DEFAULT_RULES)
    user_rules = sdk2.parameters.JSON('User rules for diff hiding', required=True, default={})
    delete_by_regex = sdk2.parameters.Bool('Delete instead of hide by regex rules',
                                           required=True, default=False)


class TasksParameters(sdk2.Task.Parameters):
    first_task = sdk2.parameters.Task('First task', task_type=AdfoxServerFunctionalShoot, required=True)
    second_task = sdk2.parameters.Task('Second task', task_type=AdfoxServerFunctionalShoot, required=True)


class AdfoxServerFunctionalShootCompare(AdfoxServerBase):
    name = "ADFOX_SERVER_FUNCTIONAL_SHOOT_COMPARE"

    class Context(sdk2.Task.Context):
        has_diff = False
        stats = None
        diff_table = None
        pg_table = None
        testenv_database = None
        report = None

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('comparison settings') as comparison_settings_group:
            comparison_settings = ComparisonParameters()

        with sdk2.parameters.Group('tasks') as tasks_group:
            tasks = TasksParameters()

    # TODO use links generating functions from yabs
    def generate_report(self):
        ok, diff = self.Context.stats.get('ok', 0), self.Context.stats.get('diff', 0)
        force_ok = self.Context.stats.get('force_ok', 0)
        total = ok + force_ok + diff

        execution_report = [
            "{0} out of {1} records have diff!".format(diff, total),
            "{0} records have 'force_ok' status!".format(force_ok),
            "<a href='{0}{1}' target='_blank'>Open aggregates table on YT</a>".format(
                'https://yt.yandex-team.ru/hahn/navigation?path=',
                self.Context.diff_table)
        ]
        if self.Context.pg_table:
            execution_report += [
                "<a href='{diff_pg_url}' target='_blank'>Show report</a>".format(
                    diff_pg_url='https://shm-reports-viewer.adfox.net/report/{0}'.format(self.Context.pg_table),
                )
            ]
        execution_report = '\n'.join(execution_report)
        self.Context.report = execution_report
        self.set_info(execution_report, do_escape=False)

    @retry(exc_cls=Exception, tries=3, delay=10)
    def process_pgaas(self, aggregates_it):
        with pgaas.get_connection(log=True) as connection:
            with connection.cursor() as cursor:
                self.Context.pg_table = str(self.id)
                create_diff_table(cursor, self.Context.pg_table,
                                  self.Context.testenv_database == constants.MAIN_TE_DATABASE)
                fill_diff_table(cursor, self.Context.pg_table, aggregates_it)

    def on_execute(self):
        task_dir = self.yt.ypath_join(constants.YT_REPORTS_DIR, str(self.id))
        self.yt.create('map_node', task_dir, recursive=True, force=False)
        self.yt.set_attribute(task_dir, constants.YT_REPORT_IS_AUTO_RUN_ATTR,
                              self.Context.testenv_database == constants.MAIN_TE_DATABASE)
        self.Context.diff_table = self.yt.ypath_join(task_dir, 'diff')

        # prepare rules
        rules_file = os.path.join(str(self.path()), 'rules_file')
        with open('rules_file', 'w') as rules_file_d:
            rules = dict()
            rules.update(self.Parameters.default_rules)
            rules.update(self.Parameters.user_rules)
            json.dump(rules, rules_file_d)

        # run comparison
        with ProcessRegistry, ProcessLog(self, logger='comparator') as process_log:
            output_file = os.path.join(str(self.path()), 'comparator.out')
            options = [
                '--proxy', constants.YT_PROXY,
                '--shoot-left', self.Parameters.first_task.Context.result_response_log,
                '--event-log-left', self.Parameters.first_task.Context.result_event_log,
                '--fraud-event-log-left', self.Parameters.first_task.Context.result_fraud_event_log,
                '--fast-fraud-event-log-left', self.Parameters.first_task.Context.result_fast_fraud_event_log,
                '--ext-req-log-left', self.Parameters.first_task.Context.result_external_requests_log,
                '--shoot-right', self.Parameters.second_task.Context.result_response_log,
                '--event-log-right', self.Parameters.second_task.Context.result_event_log,
                '--fraud-event-log-right', self.Parameters.second_task.Context.result_fraud_event_log,
                '--fast-fraud-event-log-right', self.Parameters.second_task.Context.result_fast_fraud_event_log,
                '--ext-req-log-right', self.Parameters.second_task.Context.result_external_requests_log,
                '--output-file', output_file,
                '--rules-file', rules_file,
                '--diff-table', self.Context.diff_table
            ]
            if self.Parameters.delete_by_regex:
                options += ['--delete-by-regex']
            env = dict(os.environ,
                       YT_TOKEN=get_yav_secret(constants.YAV_ROBOT_QA_ADFOX_YT_TOKEN)['YT_TOKEN'])
            with tarfile.open(str(sdk2.ResourceData(self.Parameters.comparator_executable).path)) as tar:
                tar.extractall(path=str(self.path()))
            subprocess.check_call([os.path.join(str(self.path()), 'comparator')] + options, stdout=process_log.stdout,
                                  stderr=process_log.stderr, env=env)

            # read results
            with open(output_file, 'r') as results_file:
                results = json.load(results_file)
                diff_stats = results['diff']

            self.Context.stats = diff_stats
            self.Context.has_diff = self.Context.stats.get('diff', 0) > 0

        if self.Parameters.save_results:
            aggregates_it = self.yt.read_table(self.Context.diff_table, format='json', raw=False)
            self.process_pgaas(aggregates_it)

        self.generate_report()
