import json

# Different utility constants and functions for PGaaS

# Not used, stored here for historical purposes
MAIN_TABLE_DDL = """
    create table shm.diff_tables
    (
        id serial not null,
        table_name text not null,
        created timestamp default now() not null,
        is_review boolean default true
    );

    create unique index diff_tables_id_uindex
        on shm.diff_tables (id);

    create unique index diff_tables_table_name_uindex
        on shm.diff_tables (table_name);

    alter table shm.diff_tables
        add constraint diff_tables_pk
            primary key (id);
"""

# Used for creating another diff table
DIFF_TABLE_DDL = """
    DO $$
    BEGIN
        if not exists (select 1 from pg_type where typname = 'status_type') then
            create type status_type as enum ('ok', 'force_ok', 'diff');
        end if;
    END$$;
    create table shm."{table_name}"
    (
        request_id decimal(25),
        request_data_left text,
        request_data_right text,
        request_diff text,
        request_status status_type
    );
"""

# Used for saving information about created diff table
INSERT_TABLE_INFO = """
    insert into shm.diff_tables (table_name, is_review)
    values ('{table_name}', {is_review})
"""

# Used for deleting information about old diff table
REMOVE_TABLE_INFO = """
    delete from shm.diff_tables where table_name = '{table_name}'
"""

# Used for dropping existing diff table
DROP_DIFF_TABLE = """
    DROP TABLE IF EXISTS shm."{table_name}"
"""

# Used for batch inserting rows into diff table
INSERT_DIFF_RECORDS = """
    insert into shm."{table_name}" (request_id, request_data_left, request_data_right, request_diff, request_status)
    values %s
"""

BATCH_SIZE = 50000


def create_diff_table(cursor, table_name, is_main_db):
    """
    Deletes old table and creates new table
    :param cursor:
    :param table_name:
    :param is_main_db:
    :return:
    """

    # remove table record if exists
    cursor.execute(REMOVE_TABLE_INFO.format(table_name=table_name))
    # drop table if exists
    cursor.execute(DROP_DIFF_TABLE.format(table_name=table_name))
    # create table
    cursor.execute(DIFF_TABLE_DDL.format(table_name=table_name))
    # save table info into main table
    cursor.execute(INSERT_TABLE_INFO.format(table_name=table_name, is_review='false' if is_main_db else 'true'))


def prepare_record(record):
    return [
        record['request_id'],  # request_id
        json.dumps(record['left'], indent=4, sort_keys=True),  # request_data_left
        json.dumps(record['right'], indent=4, sort_keys=True),  # request_data_right
        record['diff'],  # request_diff
        record['status']  # request_status
    ]


def fill_diff_table(cursor, table_name, aggregates_it):
    import psycopg2.extras

    insert_query = INSERT_DIFF_RECORDS.format(table_name=table_name)

    counter = 0
    records_batch = []
    for record in aggregates_it:
        if record['status'] == 'ok':  # filter out normal records to decrease space consumption
            continue
        counter += 1
        formatted_record = prepare_record(record)
        records_batch.append(formatted_record)
        if counter > BATCH_SIZE:
            psycopg2.extras.execute_values(cursor, insert_query, records_batch)
            records_batch = []
            counter = 0
    psycopg2.extras.execute_values(cursor, insert_query, records_batch)
